/*
 * On demand loader for data files
 *
 * Copyright (C) 2003--2012  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "loader.h"
#include "cmdline.h"

#include <ept/apt/apt.h>
#include <ept/debtags/debtags.h>
#include <ept/debtags/vocabulary.h>

using namespace std;

// Initialize the environment with default values
Loader::Loader() throw ()
    : m_apt(0), m_debtags(0), m_vocabulary(0), _editable(false) {}

void Loader::editable(bool val)
{
    _editable = val;
}

/// Access the apt data provider
ept::apt::Apt& Loader::apt()
{
    if (!m_apt) m_apt = new ept::apt::Apt;
    return *m_apt;
}

/// Access the debtags data provider
ept::debtags::Debtags& Loader::debtags()
{
    if (!m_debtags) m_debtags = new ept::debtags::Debtags(_editable);
    return *m_debtags;
}

/// Access the tag vocabulary
ept::debtags::Vocabulary& Loader::voc()
{
    if (!m_vocabulary) m_vocabulary = new ept::debtags::Vocabulary;
    return *m_vocabulary;
}

auto_ptr<CollPrinter> Loader::make_coll_printer(const DebtagsOptions& opts)
{
    auto_ptr<CollPrinter> printer;
    if (opts.out_names->boolValue())
        printer.reset(new CollPrinter(CollPrinter::NAME));
    else if (opts.out_facets->boolValue())
        printer.reset(new CollPrinter(CollPrinter::FACETS));
    else if (opts.out_quiet->boolValue())
        printer.reset(new CollPrinter(CollPrinter::QUIET));
    else
        printer.reset(new CollPrinter(CollPrinter::TAGS));
    return printer;
}

auto_ptr<PackagePrinter> Loader::make_package_printer(
        const DebtagsOptions& opts,
        PackagePrinter::Type default_type)
{
    auto_ptr<PackagePrinter> printer;
    if (opts.out_names->boolValue())
        printer.reset(new PackagePrinter(PackagePrinter::NAME, apt(), debtags()));
    else if (opts.out_full->boolValue())
        printer.reset(new PackagePrinter(PackagePrinter::FULL, apt(), debtags()));
    else if (opts.out_short->boolValue())
        printer.reset(new PackagePrinter(PackagePrinter::SHORT, apt(), debtags()));
    else if (opts.out_quiet->boolValue())
        printer.reset(new PackagePrinter(PackagePrinter::QUIET, apt(), debtags()));
    else
        printer.reset(new PackagePrinter(default_type, apt(), debtags()));
    return printer;
}

auto_ptr<PackagePrinter> Loader::make_package_printer(PackagePrinter::Type type)
{
    auto_ptr<PackagePrinter> printer;
    printer.reset(new PackagePrinter(type, apt(), debtags()));
    return printer;
}
