// Copyright 1999-2020 - Universit de Strasbourg/CNRS
// The Aladin Desktop program is developped by the Centre de Donnes
// astronomiques de Strasbourgs (CDS).
// The Aladin Desktop program is distributed under the terms
// of the GNU General Public License version 3.
//
//This file is part of Aladin Desktop.
//
//    Aladin Desktop is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Aladin Desktop is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    The GNU General Public License is available in COPYING file
//    along with Aladin Desktop.
//

package cds.aladin.prop;

import java.awt.Font;
import java.util.Vector;

import javax.swing.JComponent;
import javax.swing.JLabel;

/** Gre une proprit. Il s'agit d'un objet qui contient tout ce qu'il faut pour
 * afficher et ventuellement mettre  jour une proprit
 * @date dc 2011 - cration
 * @author Pierre Fernique [CDS]
 */
public class Prop {
   private String id;           // Identificateur
   private JComponent label;    // Titre (courte description)
   private JComponent widget;   // Le composant qui permet la visualisation et la mise  jour ventuelle
   private String help;         // Un help (petit paragraphe)
   private PropAction action;   // l'action  effectuer aprs une modification de la proprit par l'utilisateur (null sinon)
   private PropAction resume;   // l'action  effectuer pour rinitialiser la valeur de la proprit (null sinon)
   
   public String getId() { return id; }
   public JComponent getLabel() { return label; }
   public JComponent getWidget() { return widget; }
   public String getHelp() { return help; }
   
   /** Excute l'action lors de la mise  jour de la prioprit
    * @return ProAction.NOTHING: aucun changement,
    *         PropAction.FAILED: nouvelle valeur impossible,
    *         PropAction.SUCCESS: nouvelle valeur prise en compte
    */
   public int apply() { 
      if( action==null ) return PropAction.NOTHING;
      return action.action();
   }
   
   /** Excute la rinitialisation du widget de la proprit */
   public void resume() {
      if( resume==null ) return;
      if( widget.hasFocus() ) return; // Pour viter une maj pdt une dition
      resume.action();
   }
   
   /** Cre une proprit
    * @param id Identificateur de la proprit (unique)
    * @param label  label de la proprit (peut tre soit un String, soit directement un JComponent)
    * @param help  le help, ou null si aucun
    * @param widget  le widget associ  la prorit (peut tre un String pour du simple texte)
    * @param resumeAction  L'action  excuter lors de la rinitialisation de la proprit (ou null si aucune)
    * @param applyAction  L'action  excuter lors aprs que l'utilisateur ait mise  jour la proprit (ou null si aucune)
    * @return La proprit
    */
   static public Prop propFactory(String id,Object label,String help,Object widget,PropAction resumeAction,PropAction applyAction) {
      Prop p = new Prop();
      p.id = id;
      p.resume = resumeAction;
      JLabel lab;
      if( label instanceof String ) {
         lab = new JLabel((String)label);
         lab.setFont(lab.getFont().deriveFont(Font.BOLD));
         p.label = lab;
      } else p.label = (JComponent)label;
      p.widget = widget instanceof String ? new JLabel((String)widget) : (JComponent)widget;
      p.help = help;
      p.action = applyAction;
      return p;
   }
   
   /** Supprime une proprit d'un vecteur
    * @param propList   La liste des proprits
    * @param id         l'identificateur de la proprit  supprimer
    */
   static public void remove(Vector<Prop> propList,String id) {
      for( Prop p : propList ) {
         if( id.equals(p.getId())) { propList.remove(p); return; }
      }
   }
   
   
}


