/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.displayhash;

import android.annotation.CurrentTimeMillisLong;
import android.annotation.NonNull;
import android.graphics.Rect;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

/**
 * The verified display hash used to validate information about what was present on screen.
 */
@DataClass(genToString = true, genAidl = true)
public final class VerifiedDisplayHash implements Parcelable {
    /**
     * The timestamp when the hash was generated.
     */
    @CurrentTimeMillisLong
    private final long mTimeMillis;

    /**
     * The bounds of the requested area to generate the hash. This is in window space passed in
     * by the client.
     */
    @NonNull
    private final Rect mBoundsInWindow;

    /**
     * The selected hash algorithm that generated the image hash.
     */
    @NonNull
    private final String mHashAlgorithm;

    /**
     * The image hash generated when creating the DisplayHash.
     */
    @NonNull
    private final byte[] mImageHash;

    private String imageHashToString() {
        return byteArrayToString(mImageHash);
    }

    private String byteArrayToString(byte[] byteArray) {
        if (byteArray == null) {
            return "null";
        }
        int iMax = byteArray.length - 1;
        if (iMax == -1) {
            return "[]";
        }

        StringBuilder b = new StringBuilder();
        b.append('[');
        for (int i = 0; ; i++) {
            String formatted = String.format("%02X", byteArray[i] & 0xFF);
            b.append(formatted);
            if (i == iMax) {
                return b.append(']').toString();
            }
            b.append(", ");
        }
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/displayhash/VerifiedDisplayHash.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new VerifiedDisplayHash.
     *
     * @param timeMillis
     *   The timestamp when the hash was generated.
     * @param boundsInWindow
     *   The bounds of the requested area to generate the hash. This is in window space passed in
     *   by the client.
     * @param hashAlgorithm
     *   The selected hash algorithm that generated the image hash.
     * @param imageHash
     *   The image hash generated when creating the DisplayHash.
     */
    @DataClass.Generated.Member
    public VerifiedDisplayHash(
            @CurrentTimeMillisLong long timeMillis,
            @NonNull Rect boundsInWindow,
            @NonNull String hashAlgorithm,
            @NonNull byte[] imageHash) {
        this.mTimeMillis = timeMillis;
        com.android.internal.util.AnnotationValidations.validate(
                CurrentTimeMillisLong.class, null, mTimeMillis);
        this.mBoundsInWindow = boundsInWindow;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mBoundsInWindow);
        this.mHashAlgorithm = hashAlgorithm;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mHashAlgorithm);
        this.mImageHash = imageHash;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mImageHash);

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The timestamp when the hash was generated.
     */
    @DataClass.Generated.Member
    public @CurrentTimeMillisLong long getTimeMillis() {
        return mTimeMillis;
    }

    /**
     * The bounds of the requested area to generate the hash. This is in window space passed in
     * by the client.
     */
    @DataClass.Generated.Member
    public @NonNull Rect getBoundsInWindow() {
        return mBoundsInWindow;
    }

    /**
     * The selected hash algorithm that generated the image hash.
     */
    @DataClass.Generated.Member
    public @NonNull String getHashAlgorithm() {
        return mHashAlgorithm;
    }

    /**
     * The image hash generated when creating the DisplayHash.
     */
    @DataClass.Generated.Member
    public @NonNull byte[] getImageHash() {
        return mImageHash;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "VerifiedDisplayHash { " +
                "timeMillis = " + mTimeMillis + ", " +
                "boundsInWindow = " + mBoundsInWindow + ", " +
                "hashAlgorithm = " + mHashAlgorithm + ", " +
                "imageHash = " + imageHashToString() +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeLong(mTimeMillis);
        dest.writeTypedObject(mBoundsInWindow, flags);
        dest.writeString(mHashAlgorithm);
        dest.writeByteArray(mImageHash);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ VerifiedDisplayHash(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        long timeMillis = in.readLong();
        Rect boundsInWindow = (Rect) in.readTypedObject(Rect.CREATOR);
        String hashAlgorithm = in.readString();
        byte[] imageHash = in.createByteArray();

        this.mTimeMillis = timeMillis;
        com.android.internal.util.AnnotationValidations.validate(
                CurrentTimeMillisLong.class, null, mTimeMillis);
        this.mBoundsInWindow = boundsInWindow;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mBoundsInWindow);
        this.mHashAlgorithm = hashAlgorithm;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mHashAlgorithm);
        this.mImageHash = imageHash;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mImageHash);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<VerifiedDisplayHash> CREATOR
            = new Parcelable.Creator<VerifiedDisplayHash>() {
        @Override
        public VerifiedDisplayHash[] newArray(int size) {
            return new VerifiedDisplayHash[size];
        }

        @Override
        public VerifiedDisplayHash createFromParcel(@NonNull android.os.Parcel in) {
            return new VerifiedDisplayHash(in);
        }
    };

    @DataClass.Generated(
            time = 1617747271440L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/view/displayhash/VerifiedDisplayHash.java",
            inputSignatures = "private final @android.annotation.CurrentTimeMillisLong long mTimeMillis\nprivate final @android.annotation.NonNull android.graphics.Rect mBoundsInWindow\nprivate final @android.annotation.NonNull java.lang.String mHashAlgorithm\nprivate final @android.annotation.NonNull byte[] mImageHash\nprivate  java.lang.String imageHashToString()\nprivate  java.lang.String byteArrayToString(byte[])\nclass VerifiedDisplayHash extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genToString=true, genAidl=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
