//
// BAGEL - Parallel electron correlation program.
// Filename: _r2root_1.cc
// Copyright (C) 2013 Toru Shiozaki
//
// Author: Toru Shiozaki <shiozaki@northwestern.edu>
// Maintainer: Shiozaki group
//
// This file is part of the BAGEL package.
//
// The BAGEL package is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as published by
// the Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// The BAGEL package is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public License
// along with the BAGEL package; see COPYING.  If not, write to
// the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <algorithm>
#include <iostream>
#include <cassert>
#include <src/integral/rys/r2rootlist.h>

using namespace std;
using namespace bagel;

void R2RootList::r2root1(const double* ta, double* rr, double* ww, const int n) {

  constexpr double ax[1] = {1.000003814770027e+00};
  constexpr double aw[1] = {5.000019073704611e-01};
  constexpr double x[527] = {  8.615672004782865e-01,  9.950266140631300e-02,  1.533912849818139e-03, -5.409850309586831e-04, -1.877246401351007e-05,
  3.964916382485603e-06,  2.069710271328521e-07, -2.932418810913451e-08, -2.106101188554058e-09,  2.104829382795059e-10,  2.024247003558271e-11, -1.450167874470594e-12,
  1.244049918873936e+00,  8.692405629975766e-02, -4.119616468424005e-03, -2.428182428139501e-04,  4.052454820413477e-05, -2.365367018816754e-07, -2.939755999823473e-07,
  1.600261460738115e-08,  1.478838484787080e-09, -1.931394874416050e-10, -2.109349871930735e-12,  1.588439313333967e-12,  1.520420312449508e+00,  5.137973194761360e-02,
 -4.049376396721139e-03,  1.628573631740609e-04,  6.781166705591642e-06, -1.530730470030324e-06,  9.187187835212407e-08,  1.244819557206547e-09, -6.459780778741882e-10,
  4.659877273155007e-11, -8.503201014375361e-14, -2.658861520205036e-13,  1.673983093884788e+00,  2.704696585869289e-02, -2.097241214004720e-03,  1.331242955577535e-04,
 -5.534488526369556e-06, -4.646575658548384e-09,  2.400260672639802e-08, -2.249108425896889e-09,  1.080392014705135e-10, -5.433587247427636e-13, -3.993358516273704e-13,
  3.871128467611116e-14,  1.756755295964857e+00,  1.526939401059622e-02, -9.760889217727449e-04,  5.991237875908007e-05, -3.236265630708331e-06,  1.351713237130889e-07,
 -2.809293182807990e-09, -1.585774657376541e-10,  2.290906724458756e-11, -1.559354063446335e-12,  6.917248987963722e-14, -1.482748602858406e-15,  1.805762908789837e+00,
  9.634620173892362e-03, -4.921365250701888e-04,  2.557226571939016e-05, -1.300497736527902e-06,  6.101584375195805e-08, -2.437061621709970e-09,  7.013552343527356e-11,
 -3.488709732134984e-13, -1.286697251138654e-13,  1.120698050563011e-14, -6.192351833637347e-16,  1.837971363922831e+00,  6.646460734388877e-03, -2.784452228285442e-04,
  1.196120816031866e-05, -5.227190840898300e-07,  2.264746718662345e-08, -9.335389284735220e-10,  3.468544844880834e-11, -1.066769383557168e-12,  2.116826116756438e-14,
  2.345908610911457e-16, -5.222456556248651e-17,  1.860847534113921e+00,  4.878761611522788e-03, -1.733327430494812e-04,  6.273808634293244e-06, -2.321069795104612e-07,
  8.735947162910782e-09, -3.287693689809054e-10,  1.201478574205405e-11, -4.102661823530534e-13,  1.242329332697658e-14, -3.016824142320855e-16,  3.916841967140864e-18,
  1.877991517488670e+00,  3.741078240802150e-03, -1.156939901111019e-04,  3.621371786842781e-06, -1.152299680894067e-07,  3.738215035907146e-09, -1.233389970475859e-10,
  4.091558731011531e-12, -1.336358653188913e-13,  4.176947870763947e-15, -1.205137580327364e-16,  3.038953175999736e-18,  1.891342077656611e+00,  2.962810156580051e-03,
 -8.126455540200233e-05,  2.246728841615451e-06, -6.279795101563453e-08,  1.780912313065383e-09, -5.138321084150803e-11,  1.506868761572180e-12, -4.458313219859004e-14,
  1.310993670760591e-15, -3.750547199370709e-17,  1.015883245606674e-18,  1.902042817738832e+00,  2.405788434778285e-03, -5.934079650257259e-05,  1.471820590088685e-06,
 -3.677393684188755e-08,  9.278933628226153e-10, -2.371581684682288e-11,  6.155396758059423e-13, -1.622501974064670e-14,  4.323312946430393e-16, -1.152275862310228e-17,
  3.021040088659656e-19,  1.910815893332901e+00,  1.992941780260402e-03, -4.468319556078193e-05,  1.005936878118705e-06, -2.276381680167527e-08,  5.186082766346359e-10,
 -1.191996498974794e-11,  2.771180635960570e-13, -6.531636287848285e-15,  1.561922665212264e-16, -3.779421933054024e-18,  9.181196966451886e-20,  1.918141365457714e+00,
  1.678268301897362e-03, -3.449903529981664e-05,  7.114203879458371e-07, -1.472721058138515e-08,  3.063409501934141e-10, -6.411504411656753e-12,  1.352595342523048e-13,
 -2.882582404178766e-15,  6.219130506393346e-17, -1.359852469160702e-18,  3.008115514656167e-20,  1.924351419269268e+00,  1.432835430103866e-03, -2.719758185809975e-05,
  5.175623659526523e-07, -9.878661992941264e-09,  1.892371183315265e-10, -3.641270036396126e-12,  7.045933020294778e-14, -1.373191044537761e-15,  2.700532536716890e-17,
 -5.369532546189544e-19,  1.080387132270498e-20,  1.929683485055911e+00,  1.237670567655355e-03, -2.182402902982178e-05,  3.856251468406781e-07, -6.830347280835652e-09,
  1.213256466579220e-10, -2.162399124984232e-12,  3.870025681094534e-14, -6.961767400293297e-16,  1.260447001787078e-17, -2.300568790657252e-19,  4.238942468689443e-21,
  1.934311889610483e+00,  1.079898245253338e-03, -1.778039142701763e-05,  2.932598716763900e-07, -4.846474253439076e-09,  8.027770377019039e-11, -1.333298305458209e-12,
  2.221458339960047e-14, -3.715437572299759e-16,  6.243351805233149e-18, -1.055243727034588e-19,  1.795955182916827e-21,  1.938367570853606e+00,  9.505250333095805e-04,
 -1.467865812419493e-05,  2.270118080791165e-07, -3.516686220088063e-09,  5.458109471166947e-11, -8.489779215928949e-13,  1.323876448377117e-14, -2.070557810748222e-16,
  3.249860396672724e-18, -5.122774427508596e-20,  8.115603549956132e-22,  1.941950815969983e+00,  8.431121426034387e-04, -1.225936615946490e-05,  1.784845574279263e-07,
 -2.602250684545457e-09,  3.800064819616982e-11, -5.559279514106613e-13,  8.149724982277553e-15, -1.197570342835905e-16,  1.764672462932977e-18, -2.608863048588009e-20,
  3.871231407221051e-22,  1.945139750899207e+00,  7.529500670201870e-04, -1.034444060283336e-05,  1.422742016518369e-07, -1.959189583309629e-09,  2.701588654139113e-11,
 -3.731007628742734e-13,  5.161571431921124e-15, -7.154645316647018e-17,  9.939582286449136e-19, -1.384443249344490e-20,  1.933824234548166e-22,  1.947996151062827e+00,
  6.765289339651707e-04, -8.808853963377106e-06,  1.148080593225617e-07, -1.497919050035266e-09,  1.956665977988618e-11, -2.559260001018839e-13,  3.352337778254605e-15,
 -4.398399006477316e-17,  5.781592874390668e-19, -7.615844455382418e-21,  1.005465158942304e-22,  1.950569512007504e+00,  6.111891758621598e-04, -7.563043704663643e-06,
  9.366757264477257e-08, -1.161152375320471e-09,  1.440910031513772e-11, -1.790104293211819e-13,  2.226725006809410e-15, -2.773721784654558e-17,  3.460499586318273e-19,
 -4.324926575950356e-21,  5.415208946203335e-23,  1.952899959950286e+00,  5.548855199156916e-04, -6.541695717417760e-06,  7.718069158659645e-08, -9.113561700003159e-10,
  1.077115479510630e-11, -1.274291970815919e-13,  1.509215447294617e-15, -1.789609933458599e-17,  2.124941764669424e-19, -2.526870355654348e-21,  3.009409934088734e-23,
  1.955020370977423e+00,  5.060240979673037e-04, -5.696410269785779e-06,  6.416949759200000e-08, -7.233999767699963e-10,  8.161668098889824e-12, -9.216403853603279e-14,
  1.041745712959059e-15, -1.178744885814943e-17,  1.335308354312672e-19, -1.514604972541009e-21,  1.720194256532188e-23,  1.956957939330155e+00,  4.633478581299383e-04,
 -4.990799928148092e-06,  5.378998657957680e-08, -5.801261525846958e-10,  6.261191879127812e-12, -6.762880126009319e-14,  7.310989386852191e-16, -7.910860409705229e-18,
  8.568641199925554e-20, -9.291448636268735e-22,  1.008637920112126e-23,  1.958735355099840e+00,  4.258545603915292e-04, -4.397147623931367e-06,  4.542805043930137e-08,
 -4.696117072989436e-10,  4.857757949175649e-12, -5.028489310632612e-14,  5.209175129720071e-16, -5.400799869507380e-18,  5.604490734207370e-20, -5.821545116667268e-22,
  6.052808408504760e-24,  1.960371700407081e+00,  3.927371724872791e-04, -3.894087272300851e-06,  3.863056306775256e-08, -3.834370471210147e-10,  3.808135088677736e-12,
 -3.784471238029698e-14,  3.763518241594083e-16, -3.745436667123725e-18,  3.730411978146016e-20, -3.718658974694634e-22,  3.710056953108370e-24,  1.961883139631836e+00,
  3.633399310548020e-04, -3.464967723059611e-06,  3.305887657220832e-08, -3.155685893102470e-10,  3.013918318259026e-12, -2.880168776268112e-14,  2.754048060618046e-16,
 -2.635193064753267e-18,  2.523266104057687e-20, -2.417954411833371e-22,  2.318756090116458e-24,  1.963283456916372e+00,  3.371255301933471e-04, -3.096681487661763e-06,
  2.845690922283357e-08, -2.616239227394248e-10,  2.406462186708588e-12, -2.214659900739957e-14,  2.039282362546751e-16, -1.878916331027582e-18,  1.732273390174898e-20,
 -1.598179082218861e-22,  1.475437011379743e-24,  1.964584478992249e+00,  3.136503275906950e-04, -2.778814730158071e-06,  2.462890737578862e-08, -2.183801726113038e-10,
  1.937204593472365e-12, -1.719272593983444e-14,  1.526633644108558e-16, -1.356316033543170e-18,  1.205700651193316e-20, -1.072478936230313e-22,  9.545401129945845e-25,
  1.965796410912972e+00,  2.925454036648383e-04, -2.503022635940579e-06,  2.142372801402003e-08, -1.834398409894393e-10,  1.571342994966564e-12, -1.346598889479031e-14,
  1.154536355775585e-16, -9.903582151689067e-19,  8.499761616670204e-21, -7.299055129127475e-23,  6.271292713638548e-25,  1.966928104939505e+00,  2.735019454498620e-04,
 -2.262565062863636e-06,  1.872358391090007e-08, -1.550003468978782e-10,  1.283634394979754e-12, -1.063469262801621e-14,  8.814429358227815e-17, -7.309047030875973e-19,
  6.063693287932565e-21, -5.033120425397562e-23,  4.179708300398994e-25,  1.967987277616313e+00,  2.562598618889917e-04, -2.051957627757253e-06,  1.643588626194427e-08,
 -1.316929250646179e-10,  1.055563953559998e-12, -8.463856423178979e-15,  6.789271564580435e-17, -5.448281176399980e-19,  4.374096911557644e-21, -3.513353754092995e-23,
  2.823221281122484e-25, -2.003404801370297e+00,  2.003152656406380e-03, -3.539066651580500e-04,  6.255335668045196e-05, -1.106125933576766e-05,  1.956841700175938e-06,
 -3.463452762129637e-07,  6.132969282084482e-08, -1.086539699129032e-08,  1.925871447487880e-09, -3.412160991243461e-10,  5.869452279148779e-11, -2.001658874122280e+00,
  9.737807655580200e-04, -1.694657594663183e-04,  2.949481762449195e-05, -5.133965716887592e-06,  8.937256698578937e-07, -1.555965809516077e-07,  2.709199255085950e-08,
 -4.717665446756738e-09,  8.215766990864073e-10, -1.429682107831837e-10,  2.417020954328315e-11, -2.000819066362249e+00,  4.802915418357924e-04, -8.298414556732205e-05,
  1.433823327102506e-05, -2.477459381410711e-06,  4.280829096321190e-07, -7.397070239440100e-08,  1.278209827912003e-08, -2.208792225282748e-09,  3.816870069126800e-10,
 -6.590309889934093e-11,  1.105884214297683e-11, -2.000407000155919e+00,  2.385367502280500e-04, -4.106898228588278e-05,  7.070906850294806e-06, -1.217415510908666e-06,
  2.096066669659908e-07, -3.608892180132641e-08,  6.213628355167223e-09, -1.069840151952033e-09,  1.841977975033622e-10, -3.168786511467070e-11,  5.298910586812512e-12,
 -2.000202874036173e+00,  1.188710312477692e-04, -2.043040697711246e-05,  3.511386480002457e-06, -6.035050038464686e-07,  1.037250724584402e-07, -1.782736875812337e-08,
  3.064018410117064e-09, -5.266181377458421e-10,  9.050870358711960e-11, -1.554277014987959e-11,  2.594727336231057e-12, -2.000101281394968e+00,  5.933678311011495e-05,
 -1.018938915763951e-05,  1.749735716509465e-06, -3.004670968077623e-07,  5.159665422279360e-08, -8.860256953826178e-09,  1.521497490626098e-09, -2.612738899092531e-10,
  4.486528691038870e-11, -7.697845249922445e-12,  1.284024373682281e-12, -2.000050601901733e+00,  2.964378321837187e-05, -5.088266569908385e-06,  8.733857893892033e-07,
 -1.499140807600674e-07,  2.573230970577458e-08, -4.416875442454791e-09,  7.581437593435701e-10, -1.301330795845165e-10,  2.233641278666761e-11, -3.830750953187407e-12,
  6.387190354766723e-13, -2.000025291265628e+00,  1.481574883441851e-05, -2.542529463093782e-06,  4.363232859579300e-07, -7.487740728021597e-08,  1.284970668962736e-08,
 -2.205137325160237e-09,  3.784234728223146e-10, -6.494118297653810e-11,  1.114427665063547e-11, -1.910858880867426e-12,  3.185413682143520e-13, -2.000012643213213e+00,
  7.406339884733147e-06, -1.270864172611476e-06,  2.180693540120929e-07, -3.741882448965122e-08,  6.420748299709983e-09, -1.101745164252578e-09,  1.890499879111763e-10,
 -3.243932806469964e-11,  5.566171786406570e-12, -9.543035609160074e-13,  1.590667846591646e-13, -2.000006321001920e+00,  3.702786454246187e-06, -6.353319960747735e-07,
  1.090116188951482e-07, -1.870444609143028e-08,  3.209348761312670e-09, -5.506669071871449e-10,  9.448460087378319e-11, -1.621185762349488e-11,  2.781596239745523e-12,
 -4.768707179600799e-13,  7.948249693884887e-14, -2.000003160349815e+00,  1.851297373208257e-06, -3.176409816635553e-07,  5.450004668487049e-08, -9.350981961247989e-09,
  1.604418142504548e-09, -2.752820597864828e-10,  4.723220766508474e-11, -8.103978388823747e-12,  1.390425953804121e-12, -2.383651441975656e-13,  3.972853453881276e-14,
     0.992156368473227,      0.99608604500229,     0.998044958270205,     0.999022959494462,     0.999511599394928,     0.999755829554471,     0.999877922234645,
     0.999938962980821,     0.999969481956178,     0.999984741094518,     0.999992370576364  };
  constexpr double w[527] = {  1.196288013322608e+00, -3.316232965592209e-01,  6.095273008599551e-02, -8.299712742284355e-03,  8.942329876147542e-04,
 -7.957392659079802e-05,  6.025721136803708e-06, -3.969201067849495e-07,  2.312804303434212e-08, -1.207969269035796e-09,  5.715934179077561e-11, -2.468396282128691e-12,
  4.452346579238068e-01, -8.311524084419178e-02,  1.255931437636079e-02, -1.533468636928158e-03,  1.545039956678314e-04, -1.314963796895761e-05,  9.649744437714886e-07,
 -6.210782928571568e-08,  3.555342099219166e-09, -1.831184057722567e-10,  8.567810109678186e-12, -3.665902120277015e-13,  2.377429903237309e-01, -2.824427177117255e-02,
  3.195705818658540e-03, -3.267543641938433e-04,  2.946186579858300e-05, -2.331940320786322e-06,  1.628315706304292e-07, -1.011452876816492e-08,  5.639514172414069e-10,
 -2.846674324404023e-11,  1.310979360108809e-12, -5.538470968528684e-14,  1.589970796066651e-01, -1.287665894780264e-02,  1.056196028365156e-03, -8.507169831618671e-05,
  6.500367314683515e-06, -4.596386271478519e-07,  2.969610552777743e-08, -1.746195748464727e-09,  9.356040228991341e-11, -4.584194647047497e-12,  2.063509467676262e-13,
 -8.563095021989643e-15,  1.197385315696911e-01, -7.252709270101017e-03,  4.473192444486818e-04, -2.793183633680399e-05,  1.736370253306386e-06, -1.050449425354950e-07,
  6.056987521100858e-09, -3.280665683725062e-10,  1.655717667835332e-11, -7.761412737037370e-13,  3.378940492735505e-14, -1.366628530168849e-15,  9.626810003103103e-02,
 -4.662996063966890e-03,  2.285705503681142e-04, -1.136262179042817e-05,  5.714346901143889e-07, -2.880071604122835e-08,  1.433214365694269e-09, -6.928397498597190e-11,
  3.209072014002916e-12, -1.410417061530682e-13,  5.848668087084765e-15, -2.279340431620093e-16,  8.058351310955886e-02, -3.258637916446532e-03,  1.327174540425958e-04,
 -5.457056556711327e-06,  2.269407303202763e-07, -9.537849115198323e-09,  4.030126978263087e-10, -1.695592061691694e-11,  7.017477665482886e-13, -2.822996206092645e-14,
  1.093074789203443e-15, -4.040757643188154e-17,  6.932924008954373e-02, -2.408842598339621e-03,  8.407393338175979e-05, -2.951924168807073e-06,  1.044543675688692e-07,
 -3.730817386393375e-09,  1.345175409278384e-10, -4.882816816215734e-12,  1.773466383350940e-13, -6.388881956677985e-15,  2.260643546208645e-16, -7.777483373816629e-18,
  6.084854101958462e-02, -1.854254921468125e-03,  5.668017559358571e-05, -1.739339240190074e-06,  5.364395714930260e-08, -1.665160088791725e-09,  5.209262279241638e-11,
 -1.643283474186759e-12,  5.220593875585953e-14, -1.664097414252736e-15,  5.289649024259755e-17, -1.662420944827404e-18,  5.422395185914226e-02, -1.471866322910008e-03,
  4.004308606149555e-05, -1.092369674757146e-06,  2.990044623454687e-08, -8.219435998759474e-10,  2.271712264614489e-11, -6.319977485879579e-13,  1.771098221805933e-14,
 -4.997623669105767e-16,  1.416926816996274e-17, -4.016632699871523e-19,  4.890420128298761e-02, -1.196905181662709e-03,  2.934450313381033e-05, -7.208950817610458e-07,
  1.775267101755030e-08, -4.384630207272662e-10,  1.086909999114302e-11, -2.706711575406739e-13,  6.778109957985780e-15, -1.708171820364796e-16,  4.332642031123617e-18,
 -1.104103235705355e-19,  4.453734519471939e-02, -9.925089982409730e-04,  2.214842667432384e-05, -4.950315948637240e-07,  1.108437671661269e-08, -2.487247930122823e-10,
  5.595601387812957e-12, -1.262842492247041e-13,  2.861209771918568e-15, -6.513495672770379e-17,  1.490983203924501e-18, -3.431224548441251e-20,  4.088787282179687e-02,
 -8.364041048434408e-04,  1.712873497073616e-05, -3.512211720294877e-07,  7.211983059600051e-09, -1.483333609811154e-10,  3.056695610867962e-12, -6.313246622911625e-14,
  1.307523266529219e-15, -2.717114370594570e-17,  5.669410844132299e-19, -1.188098535847096e-20,  3.779212484611644e-02, -7.144734267524560e-04,  1.351998281198371e-05,
 -2.561030249256272e-07,  4.856836279125446e-09, -9.222643945721752e-11,  1.753878212714877e-12, -3.341076833168620e-14,  6.377471458811955e-16, -1.220276492624074e-17,
  2.341721548867187e-19, -4.507941553998703e-21,  3.513278421096276e-02, -6.174122022457515e-04,  1.085876232873705e-05, -1.911440307057374e-07,  3.367866315000303e-09,
 -5.940297912184485e-11,  1.049001231055433e-12, -1.854924666364621e-14,  3.285074543419685e-16, -5.828326505040744e-18,  1.036253137433486e-19, -1.846522448925671e-21,
  3.282353309008644e-02, -5.388831606001758e-04,  8.853138180001898e-06, -1.455521963355568e-07,  2.394906608880543e-09, -3.944036416167677e-11,  6.501538000556810e-13,
 -1.072908065467480e-14,  1.772716446194044e-16, -2.933055531212374e-18,  4.860707530205949e-20, -8.068202765847351e-22,  3.079943566852298e-02, -4.744480514700473e-04,
  7.312884660481296e-06, -1.127880866984822e-07,  1.740745387067339e-09, -2.688631126767708e-11,  4.156059104132447e-13, -6.430171359477650e-15,  9.958554583893800e-17,
 -1.544025880391745e-18,  2.396961279335339e-20, -3.725548457844600e-22,  2.901069638231954e-02, -4.209229454878210e-04,  6.110396750143155e-06, -8.875135534067295e-08,
  1.289845833267388e-09, -1.875766283776119e-11,  2.729741785762541e-13, -3.975518135189013e-15,  5.794649049380518e-17, -8.453944909166301e-19,  1.234626846441018e-20,
 -1.804754212546317e-22,  2.741848625675217e-02, -3.759753712315172e-04,  5.157883611450377e-06, -7.079349637826475e-08,  9.721647566563600e-10, -1.335758295891200e-11,
  1.836438261139916e-13, -2.526426935579217e-15,  3.478113464622101e-17, -4.791989337882007e-19,  6.607792826541135e-21, -9.118475490495710e-23,  2.599207897097340e-02,
 -3.378648199254101e-04,  4.393588495831997e-06, -5.715857851144823e-08,  7.439460960932435e-10, -9.687534699519916e-12,  1.262153489176770e-13, -1.645339056943100e-15,
  2.146161800297443e-17, -2.801273655803591e-19,  3.658981134950614e-21, -4.782243816676315e-23,  2.470684057516748e-02, -3.052710986265194e-04,  3.773204060308088e-06,
 -4.665523929846948e-08,  5.771203657811355e-10, -7.141996271054396e-12,  8.842451642015945e-14, -1.095316585130623e-15,  1.357488758214790e-17, -1.683372077920686e-19,
  2.088773528041422e-21, -2.593139222711246e-23,  2.354278983364037e-02, -2.771780960079550e-04,  3.264378020167619e-06, -3.845833802433570e-08,  4.532500071169458e-10,
 -5.343821150151194e-12,  6.302941851061969e-14, -7.437441778768584e-16,  8.780224540779231e-18, -1.037059766610395e-19,  1.225558830070594e-21, -1.448952418741844e-23,
  2.248354863560798e-02, -2.527934096691316e-04,  2.843111217496789e-06, -3.198570925690124e-08,  3.599644185586920e-10, -4.052403680512713e-12,  4.563774828297725e-14,
 -5.141665271806971e-16,  5.795116414395055e-18, -6.534481411738903e-20,  7.371635007661135e-22, -8.319161317514538e-24,  2.151556442695878e-02, -2.314917195235752e-04,
  2.491344954107831e-06, -2.681969535404320e-08,  2.888030380316357e-10, -3.110889072687120e-12,  3.352043619112596e-14, -3.613144807012200e-16,  3.896015028524903e-18,
 -4.202670051941884e-20,  4.535344284225121e-22, -4.895949151334275e-24,  2.062752577549270e-02, -2.127742115140335e-04,  2.195313279940476e-06, -2.265608135500603e-08,
  2.338780009531827e-10, -2.414994743634802e-12,  2.494432127769594e-14, -2.577287554620087e-16,  2.663773937483245e-18, -2.754123946381736e-20,  2.848592599085516e-22,
 -2.947144558781796e-24,  1.980991730138132e-02, -1.962390533430093e-04,  1.944397749045979e-06, -1.927019639452002e-08,  1.910263747709838e-10, -1.894139033776984e-12,
  1.878656032260042e-14, -1.863827035819637e-16,  1.849666309460905e-18, -1.836190342299444e-20,  1.823418127418580e-22, -1.811192649227247e-24,  1.905467666564203e-02,
 -1.815596098814495e-04,  1.730318572315303e-06, -1.649400035726463e-08,  1.572617988954484e-10, -1.499761841061657e-12,  1.430632305960189e-14, -1.365040834516150e-16,
  1.302809081919259e-18, -1.243768409439891e-20,  1.187759410109121e-22, -1.134527881679443e-24,  1.835492730745393e-02, -1.684681626705793e-04,  1.546554930153583e-06,
 -1.420033778225714e-08,  1.304131916577586e-10, -1.197947705377551e-12,  1.100656819588901e-14, -1.011505583131630e-16,  9.298048818197404e-19, -8.549246048145370e-21,
  7.862885631250444e-23, -7.233086030271359e-25,  1.770476809987225e-02, -1.567435992347740e-04,  1.387923896370533e-06, -1.229195168806714e-08,  1.088826482243708e-10,
 -9.646786056623993e-13,  8.548629578107328e-15, -7.577121120890843e-17,  6.717537847704615e-19, -5.956878939909688e-21,  5.283663226632977e-23, -4.687381572950711e-25,
  1.709910626173358e-02, -1.462020029641691e-04,  1.250270760852730e-06, -1.069371052269060e-08,  9.148066106029858e-11, -7.827260651988051e-13,  6.698433347404700e-15,
 -5.733544084409514e-17,  4.908664090940751e-19, -4.203371214687270e-21,  3.600234367557348e-23, -3.084147532402627e-25,  1.653352348837056e-02, -1.366893878232614e-04,
  1.130238654765401e-06, -9.347036956689504e-09,  7.731234114611589e-11, -6.395839024263134e-13,  5.292038621297697e-15, -4.379539001384912e-17,  3.625075364902574e-19,
 -3.001183071328613e-21,  2.485183961029660e-23, -2.058208951633333e-25,  1.600416784101971e-02, -1.280760367140562e-04,  1.025095339136016e-06, -8.205866667668458e-09,
  6.569779896950664e-11, -5.260727848395449e-13,  4.213200569210583e-15, -3.374834727194242e-17,  2.703771131874956e-19, -2.166544140037517e-21,  1.736395746043693e-23,
 -1.391839340183238e-25,  2.003329997310434e+00, -1.955654286368668e-03,  3.414054267639838e-04, -5.961283848472572e-05,  1.041124188099556e-05, -1.818699827901715e-06,
  3.177735654549567e-07, -5.553609637238462e-08,  9.708142368584593e-09, -1.697428868241392e-09,  2.966014883786106e-10, -5.034546820025400e-11,  2.001641004325865e+00,
 -9.624745873852712e-04,  1.665392731710099e-04, -2.881809571107181e-05,  4.986954032664008e-06, -8.630325519393429e-07,  1.493622934211486e-07, -2.585097988473726e-08,
  4.474404412474633e-09, -7.744711566606719e-10,  1.339463980903354e-10, -2.251298510249131e-11,  2.000814695956799e+00, -4.775310650682932e-04,  8.227526356513573e-05,
 -1.417562014954693e-05,  2.442418123076221e-06, -4.208265689095930e-07,  7.250893210796292e-08, -1.249352839088573e-08,  2.152700790405399e-09, -3.709170866812837e-10,
  6.385790872658853e-11, -1.068617001150201e-11,  2.000405919292649e+00, -2.378546057420330e-04,  4.089448569229138e-05, -7.031034005842284e-06,  1.208856951285809e-06,
 -2.078413216710355e-07,  3.573470091281851e-08, -6.143978328548772e-09,  1.056355667487168e-09, -1.816190400233468e-10,  3.120012116590073e-11, -5.210368844490053e-12,
  2.000202605264114e+00, -1.187014750040363e-04,  2.038711628167931e-05, -3.501513438799333e-06,  6.013898701608157e-07, -1.032896328165178e-07,  1.774016572918206e-08,
 -3.046905028645593e-09,  5.233113666409839e-10, -8.987755176300354e-11,  1.542362658657531e-11, -2.573138296983824e-12,  2.000101214380984e+00, -5.929451549683100e-05,
  1.017860773667812e-05, -1.747279207668338e-06,  2.999413335288833e-07, -5.148851959989753e-08,  8.838622188811205e-09, -1.517255792785942e-09,  2.604550654172260e-10,
 -4.470915094945603e-11,  7.668399446170641e-12, -1.278693536463244e-12,  2.000050585170527e+00, -2.963323143188892e-05,  5.085576344818441e-06, -8.727731221914472e-07,
  1.497830148055339e-07, -2.570536598686095e-08,  4.411487314484365e-09, -7.570878703863719e-10,  1.299293462887967e-10, -2.229758284740419e-11,  3.823431468085785e-12,
 -6.373945150974259e-13,  2.000025287085607e+00, -1.481311277341671e-05,  2.541857546446513e-06, -4.361703011065205e-07,  7.484468748527076e-08, -1.284298195714656e-08,
  2.203792852083761e-09, -3.781600647117257e-10,  6.489037059463002e-11, -1.113459452111025e-11,  1.909034221721194e-12, -3.182112547199342e-13,  2.000012642168555e+00,
 -7.405681104936618e-06,  1.270696273300405e-06, -2.180311305082709e-07,  3.741065036738835e-08, -6.419068513454907e-09,  1.101409365133129e-09, -1.889842062249958e-10,
  3.242664004588159e-11, -5.563754414054041e-12,  9.538480455599008e-13, -1.589843829253348e-13,  2.000006320740799e+00, -3.702621788713288e-06,  6.352900312219589e-07,
 -1.090020658556708e-07,  1.870240328853518e-08, -3.208928989183005e-09,  5.505829972564241e-10, -9.446816423403705e-11,  1.620868750081193e-11, -2.780992291076441e-12,
  4.767569201699482e-13, -7.946191230825729e-14,  2.000003160284540e+00, -1.851256210501093e-06,  3.176304916968217e-07, -5.449765877942659e-08,  9.350471351443169e-09,
 -1.604313221265530e-09,  2.752610872821633e-10, -4.722809960219372e-11,  8.103186093245888e-12, -1.390275015902755e-12,  2.383367048730390e-13, -3.972339035483204e-14,
   0.00392169119446278,   0.00195696227032404,  0.000977518982020638,  0.000488520018680646,  0.000244200273353903,  0.000122085219121496,  6.10388822226277e-05,
  3.05185095328528e-05,  1.52590219038026e-05,  7.62945274024323e-06,  3.81471181770676e-06  };
  int offset = -1;
  const int ibox0 = static_cast<int>(log(64.0) / log(2.0));
  for (int i = 1; i <= n; ++i) {
    double t = ta[i-1];
    offset += 1;
    if (std::isnan(t)) {
      fill_n(rr+offset, 1, 0.5);
      fill_n(ww+offset, 1, 0.0);
    } else if (t >= 131072.0) {
      for (int r = 0; r != 1; ++r) {
        ww[offset+r] = aw[1-r-1] / t;
        rr[offset+r] = 1.0 - ax[1-r-1] / t;
      }
    } else {
      assert(t >= 0);
      int it;
      double bigT = 0.0;
      if (t >= 64.0 && t < 131072.0) {
        bigT = t;
        it = static_cast<int>(log(bigT) / log(2.0) + 32 - ibox0);
        t = (t - 1.5 * pow(2.0, it + ibox0 - 32))* 2/pow(2.0, it + ibox0 - 32);
      } else {
        it = static_cast<int>(t*   0.500000000000000);
        t = (t - it *2.000000000000000-   1.000000000000000) *   1.000000000000000;
      }
      const double t2 = t * 2.0;
      for (int j=1; j <=1; ++j) {
        const int boxof = it*12+12*(j-1);
        double d = x[boxof+11];
        double e = w[boxof+11];
        double f = t2*d + x[boxof+10];
        double g = t2*e + w[boxof+10];
        d = t2*f - d + x[boxof+9];
        e = t2*g - e + w[boxof+9];
        f = t2*d - f + x[boxof+8];
        g = t2*e - g + w[boxof+8];
        d = t2*f - d + x[boxof+7];
        e = t2*g - e + w[boxof+7];
        f = t2*d - f + x[boxof+6];
        g = t2*e - g + w[boxof+6];
        d = t2*f - d + x[boxof+5];
        e = t2*g - e + w[boxof+5];
        f = t2*d - f + x[boxof+4];
        g = t2*e - g + w[boxof+4];
        d = t2*f - d + x[boxof+3];
        e = t2*g - e + w[boxof+3];
        f = t2*d - f + x[boxof+2];
        g = t2*e - g + w[boxof+2];
        d = t2*f - d + x[boxof+1];
        e = t2*g - e + w[boxof+1];
        rr[offset+j-1] = t*d - f + x[boxof+0]*0.5;
        ww[offset+j-1] = t*e - g + w[boxof+0]*0.5;
        if (64.0 <= bigT && bigT < 131072.0) {
          const int iref = 516 + (it - 32) * 1 + 1 - j;
          double rr_infty = x[iref];
          double ww_infty = w[iref];
          double Tref = pow(2.0, it + ibox0 + 1 - 32);
          ww[offset+j-1] = ww[offset+j-1] * ww_infty * Tref / bigT;
          rr[offset+j-1] = 1.0 + rr[offset+j-1] * (1.0 - rr_infty) * Tref /bigT;
        }
      }
    }
  }
}
