/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff
#include "SingleImageComponent.h"
#include "ImageComponent.h"

// -- Core stuff
#include "Slice.h"
#include "InteractiveViewer.h"
#include "Log.h"
#include "Application.h"
#include "Frame.h"

// -- VTK stuff
#include <vtkUnstructuredGrid.h>
#include <vtkImageClip.h>
#include <vtkImageChangeInformation.h>
#include <vtkProperty.h>

#include <cmath>

namespace camitk {
// -------------------- constructor  --------------------
SingleImageComponent::SingleImageComponent(Component *parentComponent, InterfaceBitMap::PossibleOrientation sliceOrientation, const QString & name, vtkSmartPointer<vtkWindowLevelLookupTable> lut)
    : Component(parentComponent, name, Component::SLICE) {

    this->sliceOrientation = sliceOrientation;

    this->lut = lut;

    // set my parent image as my parent frame
    Frame * myParentFrame = (Frame *) dynamic_cast<Frame *>(parentComponent->getFrame());
    this->setParentFrame(myParentFrame);

    // build the slice 3D
    initRepresentation();

    // by default, this slice is not shown in 3D
    this->viewSliceIn3D = false;

}

// -------------------- destructor  --------------------
SingleImageComponent::~SingleImageComponent() {
}

// -------------------- setSelected --------------------
void SingleImageComponent::setSelected(const bool b, const bool r) {
    dynamic_cast<ImageComponent*>(getParent())->setSelected(b, false);
}

// -------------------- setSelected --------------------
void SingleImageComponent::setTransformFromParent(vtkSmartPointer<vtkTransform> transform) {
    // Do nothing, my parent (the ImageComponent) should move !!!
}

// -------------------- singleImageSelected --------------------
void SingleImageComponent::singleImageSelected(const bool b) {
    Component::setSelected(b, false);
}

// -------------------- getViewSliceIn3D --------------------
bool SingleImageComponent::getViewSliceIn3D() const {
    return viewSliceIn3D;
}

// -------------------- setViewSliceIn3D --------------------
void SingleImageComponent::setViewSliceIn3D(bool toggle) {
    this->viewSliceIn3D = toggle;
    this->setVisibility(InteractiveViewer::get3DViewer(), viewSliceIn3D);
    InteractiveViewer::get3DViewer()->refresh();
}


// -------------------- initRepresentation --------------------
void SingleImageComponent::initRepresentation() {
    switch (sliceOrientation) {
    case InterfaceBitMap::AXIAL_ORIENTATION:
        mySlice = new Slice(dynamic_cast<ImageComponent *>(getParentComponent())->getImageData(), Slice::AXIAL, this->lut);
        setVisibility(InteractiveViewer::getAxialViewer(), true);

        mySlice->setImageWorldTransform(getTransformFromWorld());
        break;
    case InterfaceBitMap::CORONAL_ORIENTATION:
        mySlice = new Slice(dynamic_cast<ImageComponent *>(getParentComponent())->getImageData(), Slice::CORONAL, this->lut);
        setVisibility(InteractiveViewer::getCoronalViewer(), true);

        mySlice->setImageWorldTransform(getTransformFromWorld());
        break;
    case InterfaceBitMap::SAGITTAL_ORIENTATION:
        mySlice = new Slice(dynamic_cast<ImageComponent *>(getParentComponent())->getImageData(), Slice::SAGITTAL, this->lut);
        setVisibility(InteractiveViewer::getSagittalViewer(), true);

        mySlice->setImageWorldTransform(getTransformFromWorld());
        break;
    case InterfaceBitMap::ARBITRARY_ORIENTATION:
        mySlice = NULL;
//        setVisibility(InteractiveViewer::getArbitraryViewer(), true);
        break;
    }
}


// ---------------------- pixelPicked  ----------------------------
void SingleImageComponent::pixelPicked(double i, double j, double k) {
    ((ImageComponent*)getParent())->pixelPicked(i, j, k, this);
}

}
