/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ymonstat   ymonrange       Multi-year monthly range
      Ymonstat   ymonmin         Multi-year monthly minimum
      Ymonstat   ymonmax         Multi-year monthly maximum
      Ymonstat   ymonsum         Multi-year monthly sum
      Ymonstat   ymonmean        Multi-year monthly mean
      Ymonstat   ymonavg         Multi-year monthly average
      Ymonstat   ymonvar         Multi-year monthly variance
      Ymonstat   ymonvar1        Multi-year monthly variance [Normalize by (n-1)]
      Ymonstat   ymonstd         Multi-year monthly standard deviation
      Ymonstat   ymonstd1        Multi-year monthly standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_options.h"
#include "datetime.h"
#include "functs.h"
#include "process_int.h"


static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("ymonrange", func_range, 0, nullptr);
  cdoOperatorAdd("ymonmin",   func_min,   0, nullptr);
  cdoOperatorAdd("ymonmax",   func_max,   0, nullptr);
  cdoOperatorAdd("ymonsum",   func_sum,   0, nullptr);
  cdoOperatorAdd("ymonmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("ymonavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("ymonvar",   func_var,   0, nullptr);
  cdoOperatorAdd("ymonvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("ymonstd",   func_std,   0, nullptr);
  cdoOperatorAdd("ymonstd1",  func_std1,  0, nullptr);
  // clang-format on
}

void *
Ymonstat(void *process)
{
  constexpr int MaxMonths = 17;
  int varID;
  int nrecs;
  int levelID;
  int month_nsets[MaxMonths];
  int64_t vdates[MaxMonths];
  int vtimes[MaxMonths];
  int mon[MaxMonths];
  int nmon = 0;
  FieldVector2D vars1[MaxMonths], vars2[MaxMonths], samp1[MaxMonths];

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  const bool lrange = operfunc == func_range;
  const bool lmean = operfunc == func_mean || operfunc == func_avg;
  const bool lstd = operfunc == func_std || operfunc == func_std1;
  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  const int divisor = operfunc == func_std1 || operfunc == func_var1;
  const bool lvars2 = lvarstd || lrange;

  for (int month = 0; month < MaxMonths; month++) month_nsets[month] = 0;

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID2)) taxisDeleteBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  int tsID = 0;
  int otsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);

      if (Options::cdoVerbose) cdoPrint("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      const auto month = decodeMonth(vdate);
      if (month < 0 || month >= MaxMonths) cdoAbort("Month %d out of range!", month);

      vdates[month] = vdate;
      vtimes[month] = vtime;
      // mon[month] = vdate;

      if (!vars1[month].size())
        {
          mon[nmon++] = month;
          fieldsFromVlist(vlistID1, samp1[month], FIELD_NONE);
          fieldsFromVlist(vlistID1, vars1[month], FIELD_VEC);
          if (lvars2) fieldsFromVlist(vlistID1, vars2[month], FIELD_VEC);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          auto &rsamp1 = samp1[month][varID][levelID];
          auto &rvars1 = vars1[month][varID][levelID];

          const auto nsets = month_nsets[month];
          const auto fieldsize = rvars1.size;

          if (nsets == 0)
            {
              cdoReadRecord(streamID1, rvars1.vec.data(), &rvars1.nmiss);
              if (lrange)
                {
                  vars2[month][varID][levelID].nmiss = rvars1.nmiss;
                  vars2[month][varID][levelID].vec = rvars1.vec;
                }

              if (rvars1.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(fieldsize);

                  for (size_t i = 0; i < fieldsize; i++) rsamp1.vec[i] = !DBL_IS_EQUAL(rvars1.vec[i], rvars1.missval);
                }
            }
          else
            {
              cdoReadRecord(streamID1, field.vec.data(), &field.nmiss);
              field.size = fieldsize;
              field.grid = rvars1.grid;
              field.missval = rvars1.missval;

              if (field.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(fieldsize, nsets);

                  for (size_t i = 0; i < fieldsize; i++)
                    if (!DBL_IS_EQUAL(field.vec[i], rvars1.missval)) rsamp1.vec[i]++;
                }

              if (lvarstd)
                {
                  vfarsumq(vars2[month][varID][levelID], field);
                  vfarsum(rvars1, field);
                }
              else if (lrange)
                {
                  vfarmin(vars2[month][varID][levelID], field);
                  vfarmax(rvars1, field);
                }
              else
                {
                  vfarfun(rvars1, field, operfunc);
                }
            }
        }

      if (month_nsets[month] == 0 && lvarstd)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            vfarmoq(vars2[month][varID][levelID], vars1[month][varID][levelID]);
          }

      month_nsets[month]++;
      tsID++;
    }

  if (nmon == 12)
    {
      int smon = 0;
      for (int month = 1; month <= 12; month++)
        if (month_nsets[month]) smon++;
      if (smon == 12)
        for (int month = 1; month <= 12; month++) mon[month - 1] = month;
    }

  for (int i = 0; i < nmon; i++)
    {
      const auto month = mon[i];
      const auto nsets = month_nsets[month];
      if (nsets == 0) cdoAbort("Internal problem, nsets[%d] not defined!", month);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rsamp1 = samp1[month][varID][levelID];
          auto &rvars1 = vars1[month][varID][levelID];

          if (lmean)
            {
              if (!rsamp1.empty())
                vfardiv(rvars1, rsamp1);
              else
                vfarcdiv(rvars1, (double) nsets);
            }
          else if (lvarstd)
            {
              auto &rvars2 = vars2[month][varID][levelID];
              if (!rsamp1.empty())
                {
                  if (lstd)
                    vfarstd(rvars1, rvars2, rsamp1, divisor);
                  else
                    vfarvar(rvars1, rvars2, rsamp1, divisor);
                }
              else
                {
                  if (lstd)
                    vfarcstd(rvars1, rvars2, nsets, divisor);
                  else
                    vfarcvar(rvars1, rvars2, nsets, divisor);
                }
            }
          else if (lrange)
            {
              auto &rvars2 = vars2[month][varID][levelID];
              vfarsub(rvars1, rvars2);
            }
        }

      taxisDefVdate(taxisID2, vdates[month]);
      taxisDefVtime(taxisID2, vtimes[month]);
      cdoDefTimestep(streamID2, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rvars1 = vars1[month][varID][levelID];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1.vec.data(), rvars1.nmiss);
        }

      otsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
