/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file growisofs_interface.c
   \brief Interface to growisofs: building command line calls and calling run_command() to run them

   Fragments from man page:

\verbatim
  SYNOPSIS
      growisofs [-dry-run] [-dvd-compat] [-overburn] [-speed=1] -[Z|M] /dev/dvd <mkisofs-options>


      To master and burn an ISO9660 volume with Joliet and Rock-Ridge extensions on a DVD or Blu-ray Disc:
            growisofs -Z /dev/dvd -R -J /some/files

      To append more data to same media:
           growisofs -M /dev/dvd -R -J /more/files
      Make sure to use the same options for both initial burning and when appending data.

      To finalize the multisession DVD maintaining maximum compatibility:
           growisofs -M /dev/dvd=/dev/zero

      To use growisofs to write a pre-mastered ISO-image to a DVD:
           growisofs -dvd-compat -Z /dev/dvd=image.iso
\endverbatim



   Some notes and recording strategies (see below for sources):
   Almost all of it is copy-and-pasted from sources.

   <dvd-r[w]>
   DVD-R[W] specification defines following recording strategies:
   \li Disc-at-once
   \li Incremental sequential
   \li Restricted overwrite (DVD-RW only)

   DVD-R Dual Layer specification adds Layer Jump recording mode.

   Disk-at-once is the one claimed to guarantee full compatibility with
   DVD-ROM/-Video. DAO recordings are unappendable.

   DVD-R Single Layer Incremental strategy provides for multiple sessions

   Restricted Overwrite (DVD-RW  only) provides for arbitrary overwrite,
   meaning that it's possible to grow ISO9660 volumes within a "single
   session," in the very same way as with DVD+RW (or -RAM/plain file
   for that matter).

   Disk-at-once and Incremental recording strategies are applicable to DVD-R
   media and DVD-RW media in Sequential mode. Restricted Overwrite strategy
   is applicable to DVD-RW media explicitly formated for Restricted Overwrite.

   By default dvd-rw discs are in 'incremental sequential' format.

   A virgin DVD-RW can be directly written without the need of a formatting
   operation, however a non-virgin DVD-RW in sequential format needs to be
   blanked before to be able to write a new initial session.

   To blank a DVD-RW in(to) sequential mode, run:
   # dvd+rw-format -blank=full /dev/cd0

   To write data on a sequential DVD-RW, use the same instructions as for
   the other DVD formats:
   # growisofs -Z /dev/cd0 -J -R /path/to/data

   If you want to append some data to your previous recording, you will
   have to use the growisofs(1) -M option. However, if you perform data
   addition on a DVD-RW in incremental sequential mode, a new session will
   be created on the disc and the result will be a multi-session disc.

   A DVD-RW in restricted overwrite format does not need to be blanked
   before a new initial session, you just have to overwrite the disc with
   the -Z option, this is similar to the DVD+RW case. It is also possible
   to grow an existing ISO 9660 file system written on the disc in a same
   way as for a DVD+RW with the -M option. The result will be a one-session DVD.

   To put a DVD-RW in the restricted overwrite format, the following
   command must be used:
   # dvd+rw-format /dev/cd0

   To change back to the sequential format use:
   # dvd+rw-format -blank=full /dev/cd0
   </dvd-r[w]>

   <dvd-r>
   -dvd-compat option used with -Z option is supposed to close disc,
   making it uappendable.
   </dvd-r>

   <dvd+rw>
   dvd+rw discs need to be formatted before very first use. growisofs makes
   it itself.
   using growisofs with -M option makes existing file system 'grow' on
   target disc: there will be still only one session on disc, only larger.
   Data from new, appended batch is merged with session existing on disc.
   This is not a multisession writing.

   -dvd-compat option may be used to improve compatibility with drives, but
   it won't close disc.
   </dvd+rw>

   <multisession>
   Very few DVD-ROM drives support multisession DVDs, they will most of
   time, hopefully, only read the first session. DVD+R, DVD-R and DVD-RW
   in sequential format can accept multiple sessions, the notion of multiple
   sessions does not exist for the DVD+RW and the DVD-RW restricted
   overwrite formats.

   Using the following command after an initial (non-closed) session on a
   DVD+R, DVD-R, or DVD-RW in sequential format, will add a new session to
   the disc:
   # growisofs -M /dev/cd0 -J -R /path/to/next/data

   Using this command line with a DVD+RW or a DVD-RW in restricted
   overwrite mode, will append data in merging the new session to the
   existing one. The result will be a single-session disc. This is the
   way used to add data after an initial write on these medias.

   Note: Some space on the media is used between each session for end and
   start of sessions. Therefore, one should add sessions with large amount
   of data to optimize media space. The number of sessions is limited to
   154 for a DVD+R, about 2000 for a DVD-R, and 127 for a DVD+R Double
   Layer.
   </multisession>

   Sources:
    - http://fy.chalmers.se/~appro/linux/DVD+RW/-RW/
    - http://www.freebsd.org/doc/en/books/handbook/creating-dvds.html


   Results of my experiments with growisofs v7.1.
   First column shows intended action: first four for direct writing of
   files, last two for writing image. First row shows current format of
   disc. Body of table shows growisofs options required to get intended
   action.

   'start appendable' means: write files to empty disc and don't close the
   disc so that next batch of data can be appended

   'create single' means: write files to empty disc and close the disc so
   that it is impossible to write next batch of data (single-session)

   'continue appendable' means: write files to non-empty, appendable disc
   and don't close the disc so that next batch of data can be appended

   'write final' means: write files to non-empty, appendable disc and
   close the disc so that it is impossible to write next batch of data

   'write image (first session, appendable)' means: write iso image to empty
   disc and don't close the disc so that next batch of data can be appended

   'write image (first session, non-appen.)' means: write iso image to empty
   disc and close the disc so that it is impossible to write next batch of
   data

\verbatim
                         DVD-R              DVD+R              DVD-RW Seq.        DVD-RW Res.     DVD+R DL

start appendable         -Z                 -Z                 -Z                 -Z              -Z

create single            -dvd-compat -Z     -dvd-compat -Z     -dvd-compat -Z      n/a (1)         ? (4)

continue appendable      -M (3)             -M (3)             -M                 -M (3)          -M

write final               n/a (2)            n/a (2)           -dvd-compat -M      n/a (2)         n/a (2)

write image (first       -Z                 -Z                 -Z                 -Z               ? (4)
session, appendable)

write image (first       -dvd-compat -Z     -dvd-compat -Z     -dvd-compat -Z      n/a (1)         ? (4)
session, non-appen.)

Notes:
1. not available: using "-dvd-compat -Z" doesn't create "single-session" non-appendable disc
2. not available: using "-dvd-compat -M" doesn't close non-empty, appendable disc
3. cdw may add "-dvd-compat" option: it won't influence "appendability" of
   disc, but may improve compatibility of disc with some drives
4. no tests done yet
5. no extensive tests of "-M -dvd-compat" done yet, no external information seen yet
6. initial tests suggest no possibility of closing multisession disc


Dummy write:
                DVD-R   DVD-RW Seq   DVD-R DL Seq   DVD-R DL Jump   Other
growisofs 7.0:   y          y            y               y            n
growisofs 7.1:   y          y            y               y            n


\endverbatim
*/

#include <stdio.h>
#include <stdlib.h>

#include "cdw_string.h"
#include "gettext.h"
#include "cdw_processwin.h"
#include "cdw_thread.h"
#include "cdw_mkisofs.h"
#include "cdw_growisofs.h"
#include "cdw_logging.h"
#include "cdw_debug.h"
#include "cdw_drive.h"



static cdw_rv_t cdw_growisofs_write_from_image(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_growisofs_write_from_files(cdw_task_t *task, cdw_disc_t *disc);

static char *cdw_growisofs_create_command_write_from_image(cdw_task_t *task, cdw_disc_t *disc);
static char *cdw_growisofs_create_command_write_from_files(cdw_task_t *task, cdw_disc_t *disc);



cdw_rv_t cdw_growisofs_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id: %lld\n", task->id);

	cdw_rv_t crv = CDW_OK;
	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		crv = cdw_growisofs_write_from_image(task, disc);
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		crv = cdw_growisofs_write_from_files(task, disc);
	} else {
		cdw_assert (0, "ERROR: incorrect task id %lld\n", task->id);
	}

	return crv;
}




/**
   \brief Call growisofs to write iso image file to DVD disc

   Call one function to create command line string, and then call
   run_command() to execute this string in shell. Create processwin to
   show some progress to the user. Finish with some meaningful message
   displayed in processwin.

   \param task - variable describing current task

   \return CDW_OK if no errors occurred
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_growisofs_write_from_image(cdw_task_t *task, cdw_disc_t *disc)
{
	char *command = cdw_growisofs_create_command_write_from_image(task, disc);
	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to prepare command for writing from image\n");
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: command for writing ISO image to DVD with growisofs is:\n\"%s\"\n", command);

	/* 2TRANS: this is message in process window - there is some
	   preparation to writing, but the writing hasn't started yet */
	cdw_processwin_display_sub_info(_("Preparing to write. Please wait."));

	int rv = run_command(command, task);

	free(command);
	command = (char *) NULL;

	if (rv == 0) {
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Call growisofs to write selected files to DVD disc

   Call one function to create command line string, and then call
   run_command() to execute this string in shell. Create processwin to
   show some progress to the user. Finish with some meaningful message
   displayed in processwin.

   \param task - variable describing current task

   \return CDW_OK if no errors occurred
   \return CDW_MEM_ERROR in concat() failed
*/
cdw_rv_t cdw_growisofs_write_from_files(cdw_task_t *task, cdw_disc_t *disc)
{
	char *command = cdw_growisofs_create_command_write_from_files(task, disc);
	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to prepare command for writing from files\n");
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: command for writing files to DVD with growisofs is:\n\"%s\"\n", command);

	/* 2TRANS: this is message in process window - there is some
	   preparation to writing, but the writing hasn't started yet */
	cdw_processwin_display_sub_info(_("Preparing to write. Please wait."));

	int rv = run_command(command, task);

	free(command);
	command = (char *) NULL;

	if (rv == 0) {
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Create command line string for writing of iso image file to DVD

   The function prepares string (command) for writing of iso image file
   to DVD data disc. Memory for this string is allocated by this function
   and should be free()d by caller of this function.

   \param task - variable describing current task

   \return command string on success
   \return NULL on error
*/
char *cdw_growisofs_create_command_write_from_image(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode can't be INIT\n");
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT,
		    "ERROR: disc mode can't be INIT\n");

	bool dvd_compat = false;
	if (disc->type == CDW_DVD_RW_SEQ) {
		if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
			dvd_compat = true;
		} else {
			dvd_compat = false;
		}
	} else if (disc->type == CDW_DVD_RW_RES) {
		; /* "-dvd-compat does not influence how image is written
		     to DVD-RW Restricted, but it may improve compatibility */
		dvd_compat = true;
	} else if (disc->type == CDW_DVD_RWP) {
		; /* "-dvd-compat does not influence how image is written
		     to DVD+RW, but it may improve compatibility */
		dvd_compat = true;
	} else { /* DVD-R, DVD+R */
		if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
			dvd_compat = true;
		} else {
			dvd_compat = false;
		}
	}

	bool dao = false;
	if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
		if (task->burn.disc_mode == CDW_DISC_MODE_DAO) {
			dao = true;
		} else {
			dao = false;
		}
	} else {
		dao = false;
	}

	/* remember that when writing image to dvd no mkisofs options are
	   allowed; this includes "-use-the-force-luke=moi" */

	char speed_string[9 + 1];
	snprintf(speed_string, 9 + 1, "%lld", task->burn.speed_id);

	const char *drive = cdw_drive_get_drive_fullpath();

	cdw_assert (task->burn.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	char *command = cdw_string_concat(task->burn.tool.label,
				     " -speed=", speed_string, " ",
				     dao ? " -use-the-force-luke=dao " : " ",
				     task->burn.dummy ? " -use-the-force-luke=dummy " : " ",
				     " ", task->burn.growisofs.other_growisofs_options, " ",

				     dvd_compat ? " -dvd-compat " : " ",

				     " -Z ",
				     /* path to ISO file may contain spaces, so
					surround the path with quotes */
				     drive, "=", "\"", task->burn.iso9660_file_fullpath, "\"",

				     (char *) NULL);

	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create command string for growisofs (concat() failed)\n");
		return (char *) NULL;
	} else {
		cdw_logging_write_separator();
		/* 2TRANS: this is message printed to log file; string with
		   program name and its all command-line parameters will be
		   printed in next line */
		cdw_logging_write(_("Command for growisofs for writing from ISO image:\n"));
		cdw_logging_write("\"%s\"\n", command);

		return command;
	}
}





/**
   \brief Create command line string for direct writing of files to DVD

   The function prepares string (command) for writing data files from
   hard disc to dvd disc.

   Memory for this string is allocated by this function and should be
   free()d by caller of this function.

   The function uses prep_command_mkisofs() to create string with mkisofs
   parameters, passed to growisofs.

   \param task - variable describing current task

   \return command string on success
   \return NULL on errors
*/
char *cdw_growisofs_create_command_write_from_files(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode is not set\n");
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT,
		    "ERROR: disc mode is not set\n");

	/* 'task' already carries enough information for prep_command_mkisofs()
	   to prepare correct command string */
	char *mkisofs_options = cdw_mkisofs_create_command(task, disc);
	if (mkisofs_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to get mkisofs options string\n");
		return (char *) NULL;
	}

	bool write_initial = true; /* "-Z" will be appended if true, "-M" if false */
	bool dvd_compat = true;

	/* value of task->burn.session_mode already takes into account
	   disc type and state, e.g. it is impossible that write_mode would be
	   CDW_SESSION_MODE_START_MULTI for DVD+R that already has some data,
	   but it is be possible for DVD+RW - blank or with some initial data */
	if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI) {
		write_initial = true;

	} else if (task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {
		write_initial = false;

		if (disc->type != CDW_DVD_RW
		    && disc->type != CDW_DVD_RW_SEQ
		    && disc->type != CDW_DVD_RW_RES) {
			/* for DVD-R and DVD+R: after writing first, non-closed
			   session, adding -dvd-compat does not influence
			   'appendability' */
			dvd_compat = true;
		}
	} else if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
		write_initial = true;
		dvd_compat = true;

	} else if (task->burn.session_mode == CDW_SESSION_MODE_WRITE_FINAL) {
		write_initial = false;
		dvd_compat = true;
	}

	/* this piece of code can overwrite settings made above,
	   but it tries to set 'write_append' and 'dvd_compat' in
	   more systematic way */
	if (disc->type == CDW_DVD_RWP) {
		/* this does not influence "appendability" of DVD+RW, but
		   perhaps it will improve compatibility with some drives */
		dvd_compat = true;

		if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI) {
			write_initial = true;
		} else {
			write_initial = false;
		}

	} else if (disc->type == CDW_DVD_RW_SEQ
		   || disc->type == CDW_DVD_RW_RES) {

		if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
			write_initial = true;
			dvd_compat = true;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI) {
			write_initial = true;
			dvd_compat = false;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {
			write_initial = false;
			dvd_compat = false;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_WRITE_FINAL) {
			write_initial = false;
			dvd_compat = true;
		} else {
			;
		}
	} else if (disc->type == CDW_DVD_R
		   || disc->type == CDW_DVD_R_SEQ
		   || disc->type == CDW_DVD_RP) {
		if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {
			write_initial = true;
			dvd_compat = true;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI) {
			write_initial = true;
			dvd_compat = false;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {
			write_initial = false;
			/* this won't influence "appendability" of DVD-R or
			   DVD+R disc that already has initial session,
			   but may improve compatibility with drives */
			dvd_compat = true;
		} else if (task->burn.session_mode == CDW_SESSION_MODE_WRITE_FINAL) {
			/* this will never happen, since closing disc
			   that already has some data is a not available
			   option */
			write_initial = false;
			dvd_compat = true;
		} else {
			;
		}
	} else {
		;
	}

	bool use_dao = false;
	if (task->burn.disc_mode == CDW_DISC_MODE_DAO
	    && task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {

		use_dao = true;
	}

	cdw_logging_write_separator();
	/* 2TRANS: this is message printed to log file; string with all
	   command-line parameters passed to mkisofs will be printed in next line */
	cdw_logging_write(_("mkisofs options used by growisofs for writing files to DVD:\n"));
	cdw_logging_write("\"%s\"\n", mkisofs_options);

	char speed_string[9 + 1];
	snprintf(speed_string, 9 + 1, "%lld", task->burn.speed_id);

	const char *drive = cdw_drive_get_drive_fullpath();

	cdw_assert (task->burn.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	char *command = cdw_string_concat(task->burn.tool.label,
					  " -speed=", speed_string, " ",
					  use_dao ? " -use-the-force-luke=dao " : " ",
					  task->burn.dummy ? " -use-the-force-luke=dummy " : " ",
					  " -use-the-force-luke=moi ",

					  " ", task->burn.growisofs.other_growisofs_options, " ",

					  dvd_compat ? " -dvd-compat " : " ",

					  write_initial ? " -Z " : " -M ", drive , " ",

					  /* this includes other_mkisofs_options */
					  mkisofs_options, " ",

					  (char *) NULL);

	free(mkisofs_options);
	mkisofs_options = (char *) NULL;

	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create command string for growisofs (concat() failed)\n");
		return (char *) NULL;
	} else {
		/* 2TRANS: this is message printed to log file; string with
		   program name and its all command-line parameters will be
		   printed in next line */
		cdw_logging_write(_("Command for growisofs for writing from files:\n"));
		cdw_logging_write("\"%s\"\n", command);
		return command;
	}
}





/**
   \brief Check if growisofs supports "dummy" option for given
   type of current disc

*/
bool cdw_growisofs_allow_dummy(cdw_disc_t *disc)
{
	/* "dummy" option is supported for these
	   disc types: DVD-R, DVD-RW Seq, DVD-R DL Seq, DVD-R DL Jump

	   Currently exact types of DVD DL discs aren't recognized
	   fully, so the function returns true only for single layer discs */
	if (disc->type == CDW_DVD_R_SEQ
	    || disc->type == CDW_DVD_R) {
		return true;
	} else {
		return false;
	}
}






/* *** unused code follows *** */

#if 0

/* this function may be useful in future; disabling for now
   to suppress compiler warnings */
/**
 * \brief Create command line string for finalizing DVD disc
 *
 * The function prepares string (\p command) for finalizing DVD disc.
 *
 * Memory for this string is allocated by this function and should be
 * free()d by caller of this function. The pointer will be set to NULL if
 * function fails to allocate memory (concat() fails).
 *
 * This function uses "/dev/zero" device.
 *
 * \param command - pointer to string with prepared command
 *
 * \return CDW_OK if command created successfully
 * \return CDW_MEM_ERROR if concat() failed
 */
cdw_rv_t prep_command_growisofs_finalize(char **command)
{
	const char *ext_tool_fullpath = cdw_ext_tools_get_tool_fullpath(TOOL_GROWISOFS);
	const char *drive = cdw_drive_get_drive_fullpath();
	*command = cdw_string_concat(ext_tool_fullpath,
                          " -M ",  drive, "=/dev/zero", (char *) NULL);

	if (*command == (char *) NULL) {
		return CDW_MEM_ERROR;
	} else {
		return CDW_OK;
	}
}




/**
 * \brief Call growisofs to erase DVD disc
 *
 * Prepare and run shell command that tells growisofs to write
 * from /dev/zero file to current DVD disc. This in effect erases content
 * of dvd.
 *
 * \param task - variable describing current task
 *
 * \return CDW_MEM_ERROR in concat() failed
 * \return CDW_OK if no errors occurred
 */
cdw_rv_t run_command_growisofs_blank_dvd(cdw_task_t *task)
{
	char speed_string[10];
	snprintf(speed_string, 10, "%d", task->erase.speed);

	const char *drive = cdw_drive_get_drive_fullpath();

	const char *ext_tool_fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_GROWISOFS);
	char *command = cdw_string_concat(ext_tool_fullpath,
					  " -speed=", speed_string, " ",
					  " -Z ", drive, "=/dev/zero ",
					  (char *) NULL);
	if (command == (char *) NULL) {
		return CDW_MEM_ERROR;
	}

	/* 2TRANS: this is title of process window - program is blanking disc */
	cdw_processwin_create(_("Erase DVD"), (char *) NULL, true);
	/* 2TRANS: this is message in process window - there is some
	   preparation to erasing, but the erasing hasn't started yet */
	cdw_processwin_display_sub_info(_("Preparing to erase. Please wait."));

	int rv = run_command(command, task);

	free(command);
	command = (char *) NULL;

	if (rv == 0) {
		/* 2TRANS: this is message in process window - program finished
		   blanking disc, status unknown */
		cdw_processwin_destroy(_("Erasing of DVD finished"), true);
		return CDW_OK;
	} else {
		/* 2TRANS: this is message in process window - program didn't
		   erase blanking because of errors */
		cdw_processwin_destroy(_("Erasing of DVD failed, please see log file"), true);
		return CDW_ERROR;
	}
}



#endif

