/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ccvs.core.client;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.*;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.team.internal.ccvs.core.client.listeners.IConsoleListener;

/**
 * Class which forwards CVS console output to 1 or more
 * registered console listeners.
 */
public class ConsoleListeners implements IConsoleListener {

    private static ConsoleListeners instance = new ConsoleListeners();
    private Set listeners = new HashSet();
    
    /**
     * Return the console listeners
     * @return the console listeners
     */
    public static ConsoleListeners getInstance() {
        return instance;
    }
    
    public void addListener(IConsoleListener listener) {
        synchronized(listeners) {
            listeners.add(listener);
        }
    }
    
    public void removeListener(IConsoleListener listener) {
        synchronized(listeners) {
            listeners.remove(listener);
        }
    }

    private IConsoleListener[] getListeners() {
        synchronized(listeners) {
            return (IConsoleListener[]) listeners.toArray(new IConsoleListener[listeners.size()]);
        }
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.team.internal.ccvs.core.client.listeners.IConsoleListener#commandInvoked(java.lang.String)
     */
    public void commandInvoked(final Session session, final String line) {
        if (listeners.isEmpty()) return;
        IConsoleListener[] listeners = getListeners();
        for (int i = 0; i < listeners.length; i++) {
            final IConsoleListener listener = listeners[i];
            Platform.run(new ISafeRunnable() {
                public void handleException(Throwable exception) {
                    // Exception logged by Platform
                }
                public void run() throws Exception {
                    listener.commandInvoked(session, line);
                }
            });
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.team.internal.ccvs.core.client.listeners.IConsoleListener#messageLineReceived(java.lang.String)
     */
    public void messageLineReceived(final Session session, final String line, final IStatus status) {
        if (listeners.isEmpty()) return;
        IConsoleListener[] listeners = getListeners();
        for (int i = 0; i < listeners.length; i++) {
            final IConsoleListener listener = listeners[i];
            Platform.run(new ISafeRunnable() {
                public void handleException(Throwable exception) {
                    // Exception logged by Platform
                }
                public void run() throws Exception {
                    listener.messageLineReceived(session, line, status);
                }
            });
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.team.internal.ccvs.core.client.listeners.IConsoleListener#errorLineReceived(java.lang.String)
     */
    public void errorLineReceived(final Session session, final String line, final IStatus status) {
        if (listeners.isEmpty()) return;
        IConsoleListener[] listeners = getListeners();
        for (int i = 0; i < listeners.length; i++) {
            final IConsoleListener listener = listeners[i];
            Platform.run(new ISafeRunnable() {
                public void handleException(Throwable exception) {
                    // Exception logged by Platform
                }
                public void run() throws Exception {
                    listener.errorLineReceived(session, line, status);
                }
            });
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.team.internal.ccvs.core.client.listeners.IConsoleListener#commandCompleted(org.eclipse.core.runtime.IStatus, java.lang.Exception)
     */
    public void commandCompleted(final Session session, final IStatus status, final Exception exception) {
        if (listeners.isEmpty()) return;
        IConsoleListener[] listeners = getListeners();
        for (int i = 0; i < listeners.length; i++) {
            final IConsoleListener listener = listeners[i];
            Platform.run(new ISafeRunnable() {
                public void handleException(Throwable exception) {
                    // Exception logged by Platform
                }
                public void run() throws Exception {
                    listener.commandCompleted(session, status, exception);
                }
            });
        }
    }
}
