/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipaletteset.h"


#include "ipalette.h"
#include "ipointer.h"
#include "ipiecewisefunction.h"
#include "istring.h"


//
//  templates
//
#include "iarraytemplate.h"


IOBJECT_DEFINE_TYPE(iPaletteSet,PaletteSet,-ps,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iPaletteSet,Palette,-p,Any,0);

IOBJECT_DEFINE_KEY(iPaletteSet,Red,-r,Any,1);
IOBJECT_DEFINE_KEY(iPaletteSet,Green,-g,Any,1);
IOBJECT_DEFINE_KEY(iPaletteSet,Blue,-b,Any,1);


//
//  Helper functions hidden in a private namespace
//
namespace iPaletteSet_Private
{
	int Decode(int id, bool &r)
	{
		if(id < 0)
		{
			r = true;
			return -id - 1;
		}
		else
		{
			r = false;
			return id - 1;
		}
	}
};


using namespace iPaletteSet_Private;


//
//  Main class
//
iPaletteSet* iPaletteSet::New()
{
	return new iPaletteSet;
}


iPaletteSet::iPaletteSet() : iObject("PaletteSet")
{
	this->CreateDefaultPalettes();
}



iPaletteSet::~iPaletteSet()
{
	int i;
	for(i=0; i<mList.Size(); i++) delete mList[i];
}


iPaletteSet* iPaletteSet::Default()
{
//	static Holder self;
//	return self.Contents();
	static iPointer::AutoDeleted<iPaletteSet> self;
	return self;
}


iPalette* iPaletteSet::GetPalette(int n) const
{
	if(n>=0 && n<mList.Size()) return mList[n]; else return 0;
}


vtkLookupTable* iPaletteSet::GetLookupTable(int id) const
{
	bool r;
	int n = Decode(id,r);
	if(n>=0 && n<mList.Size()) return mList[n]->GetLookupTable(r); else return 0;
}


vtkColorTransferFunction* iPaletteSet::GetColorTransferFunction(int id) const
{
	bool r;
	int n = Decode(id,r);
	if(n>=0 && n<mList.Size()) return mList[n]->GetColorTransferFunction(r); else return 0;
}


bool iPaletteSet::AddEmptyPalette()
{
	iPalette *tmp = new iPalette; if(tmp == 0) return false;
	mList.Add(tmp);
	return true;
}


//
//  Saving and restoring the state
//
void iPaletteSet::PackStateBody(iString &s) const
{
	int i;

	//
	//  Save the palettes
	//
	this->PackValue(s,iPaletteSet::KeyPalette(),mList.Size());
	for(i=0; i<mList.Size(); i++)
	{
		this->PackValuePalette(s,mList[i]);
	}
}


void iPaletteSet::UnPackStateBody(const iString &s)
{
/*		
		if(ws == "Palette") 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			ok = true;
			if(k<0 || k>=100000) ok = false;
	
			//
			//  Do not read reversed palettes saved under CompatibilityMode=11
			//
			if(CompatibilityMode==11 && line.Section(" ",2,2).Section("/",0,0).Find("*reversed")>-1) ok = false;

			while(k>=mPalList.Size() && ok)
			{
				ok = this->AddEmptyPalette();
			}
			if(ok)
			{
				ws = line.Section(" ",2);
				if(!this->UnPackValuePalette(ws,mPalList[k])) return Error(F);
			}
		}
*/
}


void iPaletteSet::PackValuePalette(iString &s, const iPalette *p) const
{
	s = p->GetName().Substitute(" ","*") + iObjectKey::FieldSeparator();

	this->PackValuePiecewiseFunction(s,KeyPalette(),p->GetComponents(),3);
}


bool iPaletteSet::UnPackValuePalette(const iString &s, iPalette *p)
{
	bool ok = true;

	iString ws;
	ws = s.Section(iObjectKey::FieldSeparator(),0,0);
	ws.Replace("*"," ");
	if(!ws.IsEmpty()) p->SetName(ws); else ok = false;

	if(ok)
	{
		ws = s.Section(iObjectKey::FieldSeparator(),1);
		iPiecewiseFunction **c = p->GetComponents();

		ok = this->UnPackValuePiecewiseFunction(ws,KeyPalette(),c,3);

		if(!ok)
		{
			//
			//  Try the deprecated form
			//
			ok = 
				this->UnPackValuePiecewiseFunction(ws,KeyRed(),c[0]) && 
			 	this->UnPackValuePiecewiseFunction(ws,KeyGreen(),c[1]) &&
				this->UnPackValuePiecewiseFunction(ws,KeyBlue(),c[2]);
		}
		if(ok) p->Update();
	}

	return ok;
}


void iPaletteSet::CreateDefaultPalettes()
{
	iPalette* tmp;
	iPiecewiseFunction *r, *g, *b;

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Rainbow");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.1);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,0.9); b->MovePoint(1,1.0,0.9);
	r->AddPoint(1.0/6.0,0.0); g->AddPoint(1.0/6.0,0.0); b->AddPoint(1.0/6.0,1.0);
	r->AddPoint(2.0/6.0,0.0); g->AddPoint(2.0/6.0,1.0); b->AddPoint(2.0/6.0,1.0);
	r->AddPoint(3.0/6.0,0.0); g->AddPoint(3.0/6.0,1.0); b->AddPoint(3.0/6.0,0.0);
	r->AddPoint(4.0/6.0,1.0); g->AddPoint(4.0/6.0,1.0); b->AddPoint(4.0/6.0,0.0);
	r->AddPoint(5.0/6.0,1.0); g->AddPoint(5.0/6.0,0.0); b->AddPoint(5.0/6.0,0.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Temperature");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.3); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,0.8); b->MovePoint(1,1.0,0.8);
	r->AddPoint(0.7,1.0); g->AddPoint(0.6,0.0); b->AddPoint(0.8,0.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Greyscale");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,1.0); b->MovePoint(1,1.0,1.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Blue-white");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.1);
	r->MovePoint(1,1.0,0.9); g->MovePoint(1,1.0,0.9); b->MovePoint(1,1.0,1.0);
	r->AddPoint(0.75,0.0); g->AddPoint(0.38,0.0); b->AddPoint(0.78,1.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Prizm");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,0.0); g->MovePoint(1,1.0,0.0); b->MovePoint(1,1.0,0.0);
	r->AddPoint(0.25,1.0); g->AddPoint(0.25,0.0); b->AddPoint(0.25,0.0);
	r->AddPoint(0.50,0.0); g->AddPoint(0.50,1.0); b->AddPoint(0.50,0.0);
	r->AddPoint(0.75,0.0); g->AddPoint(0.75,0.0); b->AddPoint(0.75,1.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Green-white");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.1); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,1.0); b->MovePoint(1,1.0,1.0);
	r->AddPoint(0.375,0.0); 
	b->AddPoint(0.750,0.0); 
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Blue-red");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,0.0); b->MovePoint(1,1.0,0.0);
	r->AddPoint(0.25,0.0); g->AddPoint(0.25,1.0); b->AddPoint(0.25,1.0);
	r->AddPoint(0.5,0.0); g->AddPoint(0.5,0.0); b->AddPoint(0.5,1.0);
	r->AddPoint(0.75,1.0); g->AddPoint(0.75,0.0); b->AddPoint(0.75,1.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Stern");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,1.0); b->MovePoint(1,1.0,1.0);
	r->AddPoint(0.05,1.0);
	r->AddPoint(0.25,0.0);
	b->AddPoint(0.50,1.0);
	b->AddPoint(0.75,0.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Haze");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,1.0); g->MovePoint(0,0.0,0.8); b->MovePoint(0,0.0,1.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,0.8); b->MovePoint(1,1.0,0.0);
	r->AddPoint(0.5,0.0); g->AddPoint(0.5,0.1); b->AddPoint(0.5,0.5);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("Starlight");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,0.5); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,1.0); g->MovePoint(1,1.0,1.0); b->MovePoint(1,1.0,0.7);
	r->AddPoint(0.5,0.9); g->AddPoint(0.5,0.7); b->AddPoint(0.5,0.2);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("3 color");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,1.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,0.0); g->MovePoint(1,1.0,0.0); b->MovePoint(1,1.0,1.0);
	r->AddPoint(0.3333,1.0); g->AddPoint(0.3333,0.0); b->AddPoint(0.3333,0.0);
	r->AddPoint(0.3334,0.0); g->AddPoint(0.3334,1.0); b->AddPoint(0.3334,0.0);
	r->AddPoint(0.6666,0.0); g->AddPoint(0.6666,1.0); b->AddPoint(0.6666,0.0);
	r->AddPoint(0.6667,0.0); g->AddPoint(0.6667,0.0); b->AddPoint(0.6667,1.0);
	tmp->Update();
	mList.Add(tmp);

	tmp = new iPalette; IERROR_ASSERT(tmp);
	tmp->SetName("4 color");
	r = tmp->GetComponents()[0];
	g = tmp->GetComponents()[1];
	b = tmp->GetComponents()[2];
	r->MovePoint(0,0.0,1.0); g->MovePoint(0,0.0,0.0); b->MovePoint(0,0.0,0.0);
	r->MovePoint(1,1.0,0.0); g->MovePoint(1,1.0,0.0); b->MovePoint(1,1.0,1.0);
	r->AddPoint(0.249,1.0); g->AddPoint(0.249,0.0); b->AddPoint(0.249,0.0);
	r->AddPoint(0.250,1.0); g->AddPoint(0.250,1.0); b->AddPoint(0.250,0.0);
	r->AddPoint(0.499,1.0); g->AddPoint(0.499,1.0); b->AddPoint(0.499,0.0);
	r->AddPoint(0.500,0.0); g->AddPoint(0.500,1.0); b->AddPoint(0.500,0.0);
	r->AddPoint(0.749,0.0); g->AddPoint(0.749,1.0); b->AddPoint(0.749,0.0);
	r->AddPoint(0.750,0.0); g->AddPoint(0.750,0.0); b->AddPoint(0.750,1.0);
	tmp->Update();
	mList.Add(tmp);
}

