/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "crypto.h"
#include "tool.h"

#include "maintainer_record.h"

using namespace Form;

Entry MaintainerRecord::loadEntry( const QString &id, PagePolicy::State )
{
  QString filter( "objectClass=*" );
  QLdapResponse response = Connection::self()->search( id, QLdap::Base, filter );

  if ( !response.isValid() || response.entries().isEmpty() )
    return Entry();

  Entry entry = Entry::fromLdapEntry( response.entries().first() );

  if ( entry.value( "givenName" ).isEmpty() ) {
    const QString cn = entry.value( "cn" );
    const QString sn = entry.value( "sn" );

    const QString givenName = cn.left( cn.length() - sn.length() ).trimmed();
    entry.setValue( "givenName", givenName );
  }

  return entry;
}

bool MaintainerRecord::saveEntry( const Entry &entry, PagePolicy::State state, QString &errorMsg )
{
  if ( state == PagePolicy::Add || state == PagePolicy::Modify ) {

    QString dnRoot = QString( "cn=internal,%1" ).arg( Connection::self()->baseDn() );

    QLdapEntry ldapEntry;

    ldapEntry.addValue( "objectClass", "top" );
    ldapEntry.addValue( "objectClass", "inetOrgPerson" );
    ldapEntry.addValue( "objectClass", "kolabInetOrgPerson" );

    ldapEntry.setValue( "givenName", entry.value( "givenName" ) );
    ldapEntry.setValue( "sn", entry.value( "sn" ) );
    ldapEntry.setValue( "cn", QString( "%1 %2" ).arg( entry.value( "givenName" ), entry.value( "sn" ) ) );

    if ( !entry.value( "password" ).isEmpty() ) {
      QByteArray data = "{sha}" + Crypto::sha1( entry.value( "password" ).toUtf8() ).toBase64();
      ldapEntry.setValue( "userPassword", QString::fromUtf8( data ) );
    } else {
      ldapEntry.setValue( "userPassword", entry.value( "userPassword" ) );
    }

    ldapEntry.setValue( "uid", entry.value( "uid" ).toLower() );

    if ( state == PagePolicy::Modify ) {
      QString newDn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      /**
       * The 'cn' of the maintainer has changed, so we have to perform some
       * extra checks.
       */
      if ( entry.id() != newDn ) {
        /**
         * Add the object under the new dn...
         */
        if ( !Connection::self()->add( newDn, ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify maintainer '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

        /**
         * ... and remove the object under the old dn.
         */
        if ( !Connection::self()->remove( entry.id() ) ) {
          errorMsg = QObject::tr( "Could not modify maintainer '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

        /**
         * Add the new maintainer dn to the maintainer group...
         */
        QString maintainerGroupDn = QString( "cn=maintainer,%1" ).arg( dnRoot );
        QLdapEntry memberEntry;

        memberEntry.setValue( "member", newDn );
        if ( !Connection::self()->addAttributes( maintainerGroupDn, memberEntry ) ) {
          errorMsg = QObject::tr( "Could not modify maintainer '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

        /**
         * ... and remove the old maintainer dn from the maintainer group.
         */
        memberEntry.setValue( "member", entry.id() );
        if ( !Connection::self()->removeAttributes( maintainerGroupDn, memberEntry ) ) {
          errorMsg = QObject::tr( "Could not modify maintainer '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }
      } else {
        if ( !Connection::self()->modifyAttributes( entry.id(), ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify maintainer '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }
      }
    } else if ( state == PagePolicy::Add ) {

      QString dn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      /**
       * First add the maintainer...
       */
      if ( !Connection::self()->add( dn, ldapEntry ) ) {
        errorMsg = QObject::tr( "Could not add maintainer '%1': %2" )
                              .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
        return false;
      }

      /**
       * ... then add the maintainer dn to the maintainer group.
       */
      QLdapEntry memberEntry;
      memberEntry.setDn( QString( "cn=maintainer,cn=internal,%1" ).arg( Connection::self()->baseDn() ) );
      memberEntry.setValue( "member", dn );
      if ( !Connection::self()->addAttributes( memberEntry.dn(), memberEntry ) ) {
        errorMsg = QObject::tr( "Could not add maintainer '%1' to group '%2': %3" )
                      .arg( entry.value( "cn" ), memberEntry.dn(), Connection::self()->errorString() );
        return false;
      }
    }
  } else if ( state == PagePolicy::Delete ) {
    /**
     * First remove maintainer dn from maintainer group...
     */
    QLdapEntry memberEntry;
    memberEntry.setDn( QString( "cn=maintainer,cn=internal,%1" ).arg( Connection::self()->baseDn() ) );
    memberEntry.setValue( "member", entry.id() );
    if ( !Connection::self()->removeAttributes( memberEntry.dn(), memberEntry ) ) {
      errorMsg = QObject::tr( "Could not remove maintainer '%1' from group '%2': %3" )
                    .arg( entry.value( "cn" ), memberEntry.dn(), Connection::self()->errorString() );
      return false;
    }

    /**
     * ... then remove the maintainer itself.
     */
    QString ldapError;
    if ( !Tool::deleteObject( Connection::self(), entry.id(), ldapError ) ) {
      errorMsg = QObject::tr( "Could not remove maintainer '%1': %2" )
                    .arg( entry.value( "cn" ), ldapError );
      return false;
    }
  }

  return true;
}
