package latexDraw.figures;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Shape;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawException;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;

import org.sourceforge.jlibeps.epsgraphics.EpsGraphics2D;


/**
 * This class defines a kind of figure: a picture.<br>
 * <br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 * <br>
 * LaTeXDraw is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.<br>
 * <br>
 * LaTeXDraw is distributed without any warranty; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.<br>
 * <br>
 * 02/03/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Picture extends Figure
{
	private static final long serialVersionUID = 1L;

	/** The position of the picture */
	protected LaTeXDrawPoint2D position;

	/** The picture */
	protected transient Image picture;

	/** The path of the picture */
	protected String pathTarget;

	/** The path of the source picture */
	protected String pathSource;


	
	public Picture(boolean increaseMeter)
	{
		super(increaseMeter);
		
		canHaveShadow 		= false;
		isBordersMovable 	= false;
		isCustomizable 		= false;
		isDashableOrDotable 	= false;
		isDoubleBoundaryable 	= false;
		isResizable 		= false;
		isThicknessable 	= false;
		canBeFilled 		= false;
		position 			= new LaTeXDrawPoint2D();
		borders 			= new LaTeXDrawRectangle(false);
	}

	

	/**
	 * The constructor by default
	 * @param increaseMeter True: the figure will increase the number of figures
	 * @param pathSource The path of the source picture.
	 */
	public Picture(boolean increaseMeter, String pathSource)
	{
		this(increaseMeter, new LaTeXDrawPoint2D(), pathSource);
	}



	/**
	 * The constructor taking the position of the picture
	 * @param increaseMeter True: the figure will increase the number of figures
	 * @param p The position of the figure
	 * @param pathSource The path of the source picture
	 */
	public Picture(boolean increaseMeter, LaTeXDrawPoint2D p, String pathSource)
	{
		this(increaseMeter);

		if(pathSource==null)
			throw new IllegalArgumentException();
		
		if(p==null)
			p = new LaTeXDrawPoint2D();
		
		position = p;

		this.pathSource = pathSource;
		picture = getImage();
		createEPSImage();
		borders = new LaTeXDrawRectangle((LaTeXDrawPoint2D)position.clone(), new LaTeXDrawPoint2D(
						position.x+picture.getWidth(null), position.x+picture.getHeight(null)), false);
		borders.setIsFilled(true);
		updateShape();
	}



	@Override
	protected void finalize() throws Throwable
	{
		picture.flush();
		
		super.finalize();
	}



	/**
	 * @return An AWT image from the source path or null.
	 * @since 2.0.0
	 */
	public Image getImage()
	{
		return pathSource==null ? null : LaTeXDrawFrame.convertFrame.getImage(new File(pathSource));
	}
	
	
	
	/**
	 * Creates an EPS.
	 * @throws IllegalArgumentException If the writing is not possible.
	 * @since 2.0.0
	 */
	public void createEPSImage()
	{
		if(pathSource==null || picture==null)
			return ;
		
		int indexName = pathSource.lastIndexOf(File.separator)+1;
		String name = pathSource.substring(indexName, pathSource.lastIndexOf('.'))+LaTeXDrawResources.EPS_EXTENSION;
		String dirPath = pathSource.substring(0, indexName);
		pathTarget = dirPath+name;
		File file = new File(pathTarget);
		boolean created;
		
		try {// We create the output file that will contains the eps picture.
			created = file.createNewFile();
		}catch(IOException ex) { created = false; }
		
		// If created is false, it may mean that the file already exist.
		if(!created && !file.canWrite())
		{
			pathTarget = LaTeXDrawFrame.getPathExport()+File.separator+name;
			file = new File(pathTarget);
		}
		
		try
		{
			// Within jlibeps, graphics are defined using 72 DPI (72/2.54=28,3465 PPC), but latexdraw uses 50 PPC.
			// That's why, we need the scale the graphics to have a 50 PPC eps picture.
			double scale = 72. / PSTricksConstants.INCH_VAL_CM / Figure.PPC;// 72 DPI / 2.54 / 50 PPC
			FileOutputStream finalImage = new FileOutputStream(file);
			EpsGraphics2D g = new EpsGraphics2D("LaTeXDrawPicture", finalImage, 0, 0, //$NON-NLS-1$
									(int)(picture.getWidth(null)*scale), (int)(picture.getHeight(null)*scale));
			
			g.scale(scale, scale);
			g.drawImage(picture, 0, 0, null);
			g.flush();
			g.close();
			finalImage.close();
		}
		catch(Exception e) { throw new IllegalArgumentException(); }
	}


	

	@Override
	public Shape createShape2D()
	{
		if(borders==null)
			return null;
		return borders.shape;
	}




	@Override
	public Shape createNonRotatedShape2D()
	{
		if(borders==null)
			return null;
		return borders.createNonRotatedShape2D();
	}




	@Override
	public void onDragged(Point formerPt, Point newPt) throws Exception
	{
		if(formerPt.equals(newPt))
			return;

		if(isOnRotation&&borders.dSelected!=null)
		{
			borders.onDragged(formerPt, newPt);
			rotationAngle = borders.getRotationAngle();
		}
		else
			shift(formerPt, newPt);
		
		updateShape();
	}




	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx-Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx+Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x)-Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x)+Math.cos(-rotationAngle)*(cy-c2y);

		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.rotate(rotationAngle);
			g.translate(c3x, c3y);
		}

	    g.drawImage(picture, (int)position.x, (int)position.y, null);

		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}

		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}




	@Override
	@Deprecated
	public void setLastPoint(double x, double y)
	{
		// empty method
	}



	@Override
	@Deprecated
	public void setFirstPoint(double x, double y)
	{
		// empty method
	}




	@Override
	public void shift(double shiftX, double shiftY)
	{
		if(position!=null)
		{
			position.x += shiftX;
			position.y += shiftY;
		}
		
		if(borders!=null)
			borders.shift(shiftX, shiftY);
		
		updateShape();
	}




	@Override
	@Deprecated
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound)
	{
		// not rescalable
	}




	@Override
	@Deprecated
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound)
	{
		// not rescalable
	}




	@Override
	public boolean isIn(LaTeXDrawPoint2D pt)
	{
		if(borders==null)
			return false;
		
		return borders.isIn(pt);
	}




	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		if(picture==null)
			return null;

		double threshold = 0.001;
		String addBegin = "", addEnd = ""; //$NON-NLS-1$ //$NON-NLS-2$
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();
		float x = LaTeXDrawNumber.getCutNumber((float)((position.x+(SE.x-NW.x)/2.-d.x)/ppc),threshold);
		float y = LaTeXDrawNumber.getCutNumber((float)(((d.y-position.y-(SE.y-NW.y)/2.)/ppc)),threshold);

		if(rotationAngle%(Math.PI*2)!=0.)
		{
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x-d.x)/ppc;
			double cy = (d.y-gravityCenter.y)/ppc;
			double x2 = -Math.cos(-rotationAngle)*cx+Math.sin(-rotationAngle)*cy+cx;
			double y2 = -Math.sin(-rotationAngle)*cx-Math.cos(-rotationAngle)*cy+cy;
			addBegin += "\\rput{"+(float)angle+"}("+LaTeXDrawNumber.getCutNumber((float)x2,threshold)+','+LaTeXDrawNumber.getCutNumber((float)y2,threshold)+"){"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			addEnd = "}"; //$NON-NLS-1$
		}

		String path2 = pathTarget;
		path2 = path2.replaceAll("\\\\", "/");//$NON-NLS-1$ //$NON-NLS-2$

		if(path2.contains(" "))//$NON-NLS-1$
			addBegin += LaTeXDrawLang.getString1_6("Picture.0") //$NON-NLS-1$
					+System.getProperty("line.separator").charAt(0);//$NON-NLS-1$

		return addBegin+"\\rput("+x+","+y+"){"+ //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				"\\includegraphics{"+path2+"}}"+addEnd; //$NON-NLS-1$ //$NON-NLS-2$ 
	}




	/**
	 * @return the picture
	 */
	public synchronized Image getPicture()
	{
		return picture;
	}




	/**
	 * @param picture the picture to set
	 */
	public synchronized void setPicture(Image picture)
	{
		if(picture!=null)
			this.picture = picture;
	}




	/**
	 * @return the position
	 */
	public synchronized LaTeXDrawPoint2D getPosition()
	{
		return position;
	}




	/**
	 * @param position the position to set
	 */
	public synchronized void setPosition(LaTeXDrawPoint2D position)
	{
		if(position!=null)
			this.position = position;
	}




	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Picture p = (Picture)super.clone();
		p.borders = (LaTeXDrawRectangle)borders.clone();
		p.position = (LaTeXDrawPoint2D)position.clone();
		p.pathSource = pathSource;
		p.pathTarget = pathTarget;
		p.updateShape();

		return p;
	}




	/**
	 * @return the path of the target picture.
	 */
	public synchronized String getPathTarget()
	{
		return pathTarget;
	}




	/**
	 * @param path the path of the target picture to set.
	 */
	public synchronized void setPathTarget(String path)
	{
		this.pathTarget = path;
	}



	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException, LaTeXDrawException
	{
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		borders = (LaTeXDrawRectangle)ois.readObject();
		position = (LaTeXDrawPoint2D)ois.readObject();
		pathTarget = (String)ois.readObject();
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.9.5")>=0)//$NON-NLS-1$
			pathSource = (String)ois.readObject();
		else
			pathSource = pathTarget;
		
		gravityCenter = borders.getGravityCenter();
		
		try
		{
			picture = LaTeXDrawFrame.convertFrame.getImage(new File(pathSource));
			
		    if(picture==null || picture.getWidth(null)<1 || picture.getHeight(null)<1)
		    	throw new LaTeXDrawException(LaTeXDrawException.INVALID_PICTURE);
		}
		catch(Exception e)
		{
			throw new LaTeXDrawException(LaTeXDrawException.INVALID_PICTURE);
		}
	}




	@Override
	public void updateShape()
	{
		updateGravityCenter();
		shape = createShape2D();
	}




	@Override
	public boolean isTooSmallToBeRescaled()
	{
		return true;
	}




	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D origin)
	{// Fixes #1564477
		borders.mirrorHorizontal(origin);
		position.setLocation(borders.getTheNWPoint().x, position.y);
		updateShape();
	}




	@Override
	public void mirrorVertical(LaTeXDrawPoint2D origin)
	{// Fixes #1564477
		borders.mirrorVertical(origin);
		position.setLocation(position.x, borders.getTheNWPoint().y);
		updateShape();
	}




	@Override
	@Deprecated
	public LaTeXDrawPoint2D getLastPoint()
	{
		return null;
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		position.setLocation(grid.getTransformedPointToGrid(position, false));
	}




	@Override
	public int getSelectedDelimitorOrientation()
	{
		return DELIMITOR_ORIENTATION_NONE;
	}
	
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()+picture.hashCode();
	}


	
	/**
	 * @return the pathSource.
	 * @since 2.0.0
	 */
	public synchronized String getPathSource()
	{
		return pathSource;
	}




	/**
	 * @param pathSource the pathSource to set.
	 * @since 2.0.0
	 */
	public synchronized void setPathSource(String pathSource)
	{
		this.pathSource = pathSource;
	}
}
