package latexDraw.generators.svg;

import latexDraw.figures.Picture;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGImageElement;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for an picture.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LPictureSVGGenerator extends LShapeSVGGenerator
{
	public LPictureSVGGenerator(Picture f)
	{
		super(f);
	}

	
	
	/**
	 * Creates a picture from a SVGImage element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LPictureSVGGenerator(SVGImageElement elt)
	{
		this(new Picture(true));
		
		if(elt==null)
			throw new IllegalArgumentException();
		
		Picture p = (Picture)getShape();
		p.setPathSource(elt.getURI());
		p.setPicture(p.getImage());
		
		if(p.getPicture()==null)
			throw new IllegalArgumentException();
		
		p.createEPSImage();
		p.getPosition().setLocation(elt.getX(), elt.getY());
		p.getBorders().setFirstPoint((LaTeXDrawPoint2D)p.getPosition().clone());
		p.getBorders().setLastPoint(p.getPosition().x+p.getPicture().getWidth(null), p.getPosition().x+p.getPicture().getHeight(null));
		p.getBorders().setIsFilled(true);
		p.updateShape();
		applyTransformations(elt);
	}
	
	
	
	
	public LPictureSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a picture from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LPictureSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Picture(false));
		
		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGImageElement))
			throw new IllegalArgumentException();
		
		SVGImageElement main = (SVGImageElement)elt2;

		Picture p = (Picture)getShape();
		p.setPathSource(main.getURI());
		p.setPicture(p.getImage());
		
		if(p.getPicture()==null)
			throw new IllegalArgumentException();
		
		double x = main.getX();
		double y = main.getY();
		
		p.createEPSImage();
		p.getPosition().setLocation(x, y);
		p.getBorders().setFirstPoint(new LaTeXDrawPoint2D(x, y));
		p.getBorders().setLastPoint(x+p.getPicture().getWidth(null), y+p.getPicture().getHeight(null));
		p.getBorders().setIsFilled(true);
		p.updateShape();
		
		if(withTransformation)
			applyTransformations(elt);
	}


	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc == null)
			return null;

		SVGElement root = new SVGGElement(doc), img;
		Picture p = (Picture)getShape();
		
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':' + LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_PICTURE);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		
		img = new SVGImageElement(doc, p.getPathSource());
		img.setAttribute(SVGAttributes.SVG_X, String.valueOf(p.getPosition().x));
		img.setAttribute(SVGAttributes.SVG_Y, String.valueOf(p.getPosition().y));
		img.setAttribute(SVGAttributes.SVG_HEIGHT, String.valueOf(p.getPicture().getHeight(null)));
		img.setAttribute(SVGAttributes.SVG_WIDTH, String.valueOf(p.getPicture().getWidth(null)));

		root.appendChild(img);

		return root;
	}
}
