/*
 *  @(#)AutoDocJUnitListener.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.junit;

import junit.framework.Test;
import junit.framework.TestListener;
import junit.framework.AssertionFailedError;

import java.util.Vector;
import java.util.Enumeration;

import net.sourceforge.groboutils.autodoc.v1.spi.TestListenerFactory;

import net.sourceforge.groboutils.util.classes.v1.SPISingletonStore;

import org.apache.log4j.Logger;


/**
 * Listens to JUnit tests, and chains the calls to other listeners.  This
 * allows for a central repository to register a listener.  This class allows
 * for using the SPI extention mechanism to define delegated-to listeners
 * if no listeners are passed to the constructor.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 4, 2002
 * @version    $Date: 2003/02/10 22:52:12 $
 */
public class AutoDocJUnitListener implements TestListener
{
    private static final Logger LOG = Logger.getLogger( AutoDocJUnitListener.class );
    
    private static final SPISingletonStore s_factoryStore =
        new SPISingletonStore( TestListenerFactory.class,
            SPISingletonStore.NO_MULTIPLES_SILENT );
    
    
    private Vector listeners = new Vector();
    
    
    /**
     * Create a new listener, using the singleton store to specify all
     * test listener factories.
     */
    public AutoDocJUnitListener()
    {
        this( null );
    }
    
    
    /**
     * Create a new listener using the given factories to specify all
     * JUnit listeners.  If <tt>factories</tt> is <tt>null</tt>, then
     * the singleton store will be used to get all factories.
     */
    public AutoDocJUnitListener( TestListenerFactory factories[] )
    {
        if (factories == null)
        {
            factories = getFactories();
            if (factories == null)
            {
                LOG.warn( "No factories defined for "+getClass() );
                factories = new TestListenerFactory[0];
            }
        }
        
        for (int i = 0; i < factories.length; ++i)
        {
            loadListeners( factories[i] );
        }
    }
    
    
    /**
     * Add an individual JUnit TestListener to the inner set of listeners.
     *
     * @param tl the listener to register.  If <tt>tl</tt> is <tt>null</tt>,
     *      then the request is ignored.
     */
    public void addListener( TestListener tl )
    {
        if (tl != null)
        {
            this.listeners.addElement( tl );
        }
    }
    
    
    /**
     * Return a list of all known inner listeners.
     */
    public Enumeration getListeners()
    {
        return listeners.elements();
    }
    
    
    //-------------------------------------------------------------------------
    // JUnit TestListener methods
    
    
    /**
     * An error occurred.
     */
    public void addError(Test test, Throwable t)
    {
        for (Enumeration enum = getListeners(); enum.hasMoreElements();)
        {
            TestListener tl = (TestListener)enum.nextElement();
            tl.addError( test, t );
        }
    }
    
    /**
     * A failure occurred.
     */
    public void addFailure(Test test, AssertionFailedError t)
    {
        for (Enumeration enum = getListeners(); enum.hasMoreElements();)
        {
            TestListener tl = (TestListener)enum.nextElement();
            tl.addFailure( test, t );
        }
    }
    
    
    /**
     * A test ended.
     */
    public void endTest(Test test)
    {
        for (Enumeration enum = getListeners(); enum.hasMoreElements();)
        {
            TestListener tl = (TestListener)enum.nextElement();
            tl.endTest( test );
        }
    }
    
    
    /**
     * A test started.
     */
    public void startTest(Test test)
    {
        for (Enumeration enum = getListeners(); enum.hasMoreElements();)
        {
            TestListener tl = (TestListener)enum.nextElement();
            tl.startTest( test );
        }
    }
    
    
    //-------------------------------------------------------------------------
    // protected methods
    
    
    /**
     * Creates the listener from the factory, and adds it to the inner list.
     *
     * @param factory the factory to create the listener from.
     * @see #addListener( TestListener )
     */
    protected void loadListeners( TestListenerFactory factory )
    {
        if (factory != null)
        {
            TestListener tl = factory.createListener();
            addListener( tl );
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Static methods
        
    /**
     * Retrieve the AutoDocJUnitListenerFactory singleton store for setting up
     * the factory to be used for all unspecified constructor
     * invocations to AutoDocJUnitListener instances.
     */
    public static SPISingletonStore getFactoryStore()
    {
        return s_factoryStore;
    }
    
    
    /**
     * Return all the currently known test listener factories, using the
     * singleton returned by <tt>getFactoryStore()</tt>
     */
    protected static TestListenerFactory[] getFactories()
    {
        Vector v = new Vector();
        Enumeration enum = getFactoryStore().getSingletons();
        while (enum.hasMoreElements())
        {
            v.addElement( enum.nextElement() );
        }
        
        TestListenerFactory[] tlf = new TestListenerFactory[ v.size() ];
        v.copyInto( tlf );
        return tlf;
    }
}

