/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2012 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

// Smooth Tasks
#include "SmoothTasks/TaskIcon.h"
#include "SmoothTasks/TaskItem.h"
#include "SmoothTasks/Applet.h"

// Qt
#include <QPainter>
#include <QStyleOptionGraphicsItem>
#include <QFont>
#include <QApplication>
#include <QPropertyAnimation>

// Plasma
#include <Plasma/Applet>
#include <Plasma/PaintUtils>

// KDE
#include <KIcon>
#include <KIconEffect>
#include <KIconLoader>

// Std C++
#include <iterator>

namespace SmoothTasks {

TaskIcon::TaskIcon(TaskItem *item)
	: QObject(item)
	, m_item(item)
	, m_icon()
	, m_highlightColor(0)
	, m_contentsRdRect()
	, m_pixmap()
	, m_progress(0.0)
	, m_size(128.0, 128.0)
	, m_startupAnimation(0)
{}

QRgb TaskIcon::highlightColor() const {
	Applet *applet = m_item->applet();
	if (m_highlightColor != 0 && applet->lightColorFromIcon()) {
		return m_highlightColor;
	}
	else {
		return applet->lightColor().rgb();
	}
}

void TaskIcon::updatePos() {
	const qreal rdHeight = this->rdHeight();
	const QSize iconRdSize = actualRdSize(QSize(m_contentsRdRect.width(), rdHeight));
	const qreal maxRdWidth = qMin(m_contentsRdRect.width(), m_item->applet()->minIconSpace());
	const qreal shift = iconRdSize.width() < maxRdWidth ? 0.5 * (maxRdWidth - iconRdSize.width()) : 0.0;

	if (m_item->orientation() == Qt::Vertical) {
		if (QApplication::isRightToLeft()) {
			m_pos.setX(m_contentsRdRect.center().y() - 0.5 * iconRdSize.height());
			m_pos.setY(m_contentsRdRect.left() + shift);
		} else {
			m_pos.setX(m_contentsRdRect.center().y() - 0.5 * iconRdSize.height());
			m_pos.setY(m_contentsRdRect.right() - iconRdSize.width() - shift);
		}
		m_size = QSize(rdHeight, m_contentsRdRect.width());
	} else {
		if (QApplication::isRightToLeft()) {
			m_pos.setX(m_contentsRdRect.right() - iconRdSize.width() - shift);
			m_pos.setY(m_contentsRdRect.center().y() - 0.5 * iconRdSize.height());
		} else {
			m_pos.setX(m_contentsRdRect.left() + shift);
			m_pos.setY(m_contentsRdRect.center().y() - 0.5 * iconRdSize.height());
		}
		m_size = QSize(m_contentsRdRect.width(), rdHeight);
	}
}

QSizeF TaskIcon::preferredRdSize(const QIcon& qIcon, const Applet& applet, const Qt::Orientation& orientation, const qreal rdHeightContents, const qreal rdHeightMax) {
	qreal scaledRdHeight = applet.iconScale() * rdHeightContents;
	if (scaledRdHeight > rdHeightMax)
		scaledRdHeight = rdHeightMax;
	if (scaledRdHeight < 1.0)
		scaledRdHeight = 1.0;
	if (scaledRdHeight > 128.0)
		scaledRdHeight = 128.0;  // std::numeric_limits<qreal>::infinity() does not work
	QSizeF iconRdSize = actualRdSize(qIcon, orientation, QSize(128.0, scaledRdHeight));
	return QSizeF(
			qMax(iconRdSize.width(), applet.minIconSpace()),
			iconRdSize.height() / applet.iconScale()
		);
}

QSizeF TaskIcon::preferredRdSize(const qreal rdHeightContents, const qreal rdHeightMax) const {
	return preferredRdSize(m_icon, *m_item->applet(), m_item->orientation(), rdHeightContents, rdHeightMax);
}

void TaskIcon::paint(QPainter *p, qreal hover, bool isGroup) {
	m_pixmap = m_icon.pixmap(m_icon.actualSize(m_size));  // without m_icon.actualSize the jumping startup icon uses too much space on expanded buttons

	if (m_pixmap.isNull()) {
		kDebug() << "TaskIcon pixmap is null";
		return;
	}

	if (m_startupAnimation) {
		animationStartup(m_progress);
	}

	if (hover > 0.0) {
		animationHover(hover);
	}

	p->drawPixmap(m_pos, m_pixmap);
}

void TaskIcon::setRects(const QRectF &contentsRdRect, const QRectF &maxRdRect) {
	if (m_contentsRdRect != contentsRdRect || m_maxRdRect != maxRdRect) {
		m_contentsRdRect = contentsRdRect;
		m_maxRdRect = maxRdRect;
		updatePos();
	}
}

void TaskIcon::setIcon(const QIcon& icon) {
	m_icon = icon;
	m_highlightColor = dominantColor();
	updatePos();
}

qreal TaskIcon::rdHeight() const {
	QPointF center(m_contentsRdRect.center());
	return qMax(1.0, qMin(
			m_item->applet()->iconScale() * m_contentsRdRect.height(),
			2.0 * qMin(center.y() - m_maxRdRect.top(), m_maxRdRect.bottom() - center.y())
		));
}

QSize TaskIcon::actualRdSize(const QIcon& qIcon, const Qt::Orientation& orientation, const QSize &size, QIcon::Mode mode, QIcon::State state) {
	if (orientation == Qt::Vertical) {
		QSize iconSize = qIcon.actualSize(QSize(size.height(), size.width()), mode, state);
		return QSize(iconSize.height(), iconSize.width());
	} else {
		return qIcon.actualSize(size, mode, state);
	}
}

QSize TaskIcon::actualRdSize(const QSize &size, QIcon::Mode mode, QIcon::State state) const {
	return actualRdSize(m_icon, m_item->orientation(), size, mode, state);
}

QSize TaskIcon::rdSize() {
	QSize iconRdSize = actualRdSize(QSize(m_contentsRdRect.width(), this->rdHeight()));
	const qreal maxRdWidth = qMin(m_contentsRdRect.width(), m_item->applet()->minIconSpace());
	return QSize(iconRdSize.width() < maxRdWidth ? maxRdWidth : iconRdSize.width(), iconRdSize.height());
}

void TaskIcon::startStartupAnimation(int duration) {
	if (!m_startupAnimation) {
		m_startupAnimation = new QPropertyAnimation(this, "startupAnimationProgress", this);
		m_startupAnimation->setEasingCurve(QEasingCurve::Linear);
		m_startupAnimation->setEndValue(1.0);
		m_startupAnimation->setLoopCount(-1);
		m_startupAnimation->setStartValue(0.0);
	}
	m_startupAnimation->setDuration(duration);
	m_startupAnimation->start();
}

void TaskIcon::stopStartupAnimation() {
	delete m_startupAnimation;
	m_startupAnimation = 0;
}

void TaskIcon::setStartupAnimationProgress(qreal progress) {
	m_progress = progress;
	emit update();
}

void TaskIcon::animationHover(qreal hover) {
	KIconEffect *effect = KIconLoader::global()->iconEffect();

	if (effect->hasEffect(KIconLoader::Desktop, KIconLoader::ActiveState)) {
		if (qFuzzyCompare(qreal(1.0), hover)) {
			m_pixmap = effect->apply(
				m_pixmap,
				KIconLoader::Desktop,
				KIconLoader::ActiveState);
		}
		else if (hover != 0.0) {
			m_pixmap = Plasma::PaintUtils::transition(
				m_pixmap,
				effect->apply(
					m_pixmap,
					KIconLoader::Desktop,
					KIconLoader::ActiveState),
				hover);
		}
	}
}

void TaskIcon::animationStartup(qreal progress) {
	QPixmap pixmap = QPixmap(m_pixmap.width(), m_pixmap.height());
	pixmap.fill(Qt::transparent);
	int width;
	int height;

	if (progress < 0.5) {
		width  = m_pixmap.width()  * (0.5 + progress * 0.5);
		height = m_pixmap.height() * (1.0 - progress * 0.5);
	}
	else {
		width  = m_pixmap.width()  * (1.0 - progress * 0.5);
		height = m_pixmap.height() * (0.5 + progress * 0.5);
	}

	QPixmap scaled = m_pixmap.scaled(
		width, height,
		Qt::IgnoreAspectRatio,
		Qt::SmoothTransformation);

	if (!scaled.isNull()) {
		QPainter pixmapPainter(&pixmap);
		pixmapPainter.drawPixmap(
			(m_pixmap.width()  - width)  / 2,
			(m_pixmap.height() - height) / 2,
			scaled);


		pixmapPainter.end();
	}
	m_pixmap = pixmap;

	QPixmap transparentPixmap = QPixmap(m_pixmap.width(), m_pixmap.height());
	transparentPixmap.fill(Qt::transparent);
	m_pixmap = Plasma::PaintUtils::transition(transparentPixmap, m_pixmap, 0.85);
}

QRgb TaskIcon::averageColor() const {
	// Computes, and returns average color of the icon image.
	// Added by harsh@harshj.com for color hot-tracking support.
	QImage image(m_icon.pixmap(m_size).toImage());
	unsigned int r(0), g(0), b(0);
	unsigned int count = 0;

	for(int x = 0; x < image.width(); ++ x) {
		for(int y = 0; y < image.height(); ++ y) {
			QRgb color = image.pixel(x, y);
			
			if (qAlpha(color) != 0) {
				r += qRed(color);
				g += qGreen(color);
				b += qBlue(color);
				++ count;
			}
		}
	}
	r /= count;
	g /= count;
	b /= count;

	return qRgb(r, g, b);
}

static bool hsvLess(const QColor& c1, const QColor& c2) {
	int h1, s1, v1, h2, s2, v2;
	c1.getHsv(&h1, &s1, &v1);
	c2.getHsv(&h2, &s2, &v2);
	
	return
		(h1 << 16 | s1 << 8 | v1) <
		(h2 << 16 | s2 << 8 | v2);
	/*
	 * should be equivalent to this, but with less branches:
	int cmp = h1 - h2;
	
	if (cmp == 0) {
		cmp = s1 - s2;
		if (cmp == 0) {
			cmp = v1 - v2;
		}
	}
	return cmp < 0;
	*/
}

QRgb TaskIcon::meanColor() const {
	QImage image(m_icon.pixmap(m_size).toImage());
	QVector<QColor> colors(image.width() * image.height());
	
	int count = 0;
	
	for(int x = 0; x < image.width(); ++ x) {
		for(int y = 0; y < image.height(); ++ y) {
			QRgb color = image.pixel(x, y);
			
			// only use non-(total-)transparent colors
			if (qAlpha(color) != 0) {
				colors[count] = color;
				++ count;
			}
		}
	}
	
	if (count == 0) {
		return 0;
	}

	colors.resize(count);
	qSort(colors.begin(), colors.end(), hsvLess);
	
	int mid = count / 2;
	if (count & 1) { // odd case
		return colors[mid].rgb();
	}
	else { // even case
		QColor color1(colors[mid]);
		QColor color2(colors[mid + 1]);
		
		int r = (color1.red()   + color2.red())   / 2;
		int g = (color1.green() + color2.green()) / 2;
		int b = (color1.blue()  + color2.blue())  / 2;
		
		return qRgb(r, g, b);
	}
}

static bool isNear(const QColor& c1, const QColor& c2) {
	int h1, s1, v1, h2, s2, v2;
	c1.getHsv(&h1, &s1, &v1);
	c2.getHsv(&h2, &s2, &v2);
	
	return
		qAbs(h1 - h2) <=  8 &&
		qAbs(s1 - s2) <= 16 &&
		qAbs(v1 - v2) <= 32;
}

QRgb TaskIcon::dominantColor() const {
	QImage image(m_icon.pixmap(m_size).toImage());
	QVector<QColor> colors(image.width() * image.height());
	
	int count = 0;
	
	// find the mean color
	for(int x = 0; x < image.width(); ++ x) {
		for(int y = 0; y < image.height(); ++ y) {
			QRgb rgb = image.pixel(x, y);
			
			// only use non-(total-)transparent colors
			if (qAlpha(rgb) != 0) {
				QColor color(rgb);
				
				// only use colors that aren't too grey
				if (color.saturation() > 24) {
					colors[count] = color;
				
					++ count;
				}
			}
		}
	}
	
	if (count == 0) {
		return 0;
	}
	
	colors.resize(count);
	qSort(colors.begin(), colors.end(), hsvLess);
	
	int mid = count / 2;
	QColor midColor(colors[mid]);
	QColor* begin = colors.begin() + mid;
	
	// find similar colors before the mean:
	if (mid != 0) {
		-- begin;
	
		while (begin != colors.begin()) {
			if (isNear(*(begin - 1), midColor)) {
				-- begin;
			}
			else {
				break;
			}
		}
	}
	
	QColor* end = colors.begin() + mid;
	
	// find similar colors after the mean:
	while (end != colors.end()) {
		if (isNear(*end, midColor)) {
			++ end;
		}
		else {
			break;
		}
	}
	
	// average of similar colors:
	unsigned int r = 0, g = 0, b = 0;
	for (QColor* it = begin; it != end; ++ it) {
		r += it->red();
		g += it->green();
		b += it->blue();
	}
	
	int similarCount = std::distance(begin, end);
	QColor color(r / similarCount, g / similarCount, b / similarCount);
	int h, s, v;
	color.getHsv(&h, &s, &v);
	
	if (v < 196) {
		v = 196;
	}
	
	if (s < 128) {
		s = 128;
	}
	
	color.setHsv(h, s, v);
	
	return color.rgb();
}

} // namespace SmoothTasks
#include "TaskIcon.moc"
