/* primax_scan -- linux driver for Primax scanners

   Authors:
   Marco Foglia <Marco.Foglia@switzerland.org>
   Thomas Schano <schano@t-online.de>
   Christian Ordig <chr.ordig@gmx.net>

   Copyright (C) 1999 Marco Foglia, Thomas Schano, Christian Ordig

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/        

#include <asm/io.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>            
#include <sched.h>
#include <sys/time.h>
#include <tiffio.h>
#include <getopt.h>
                        
#include "primax_scan.h"
#include "lp.h"
#include "tables.h"
#include "LM9811.h"

int dbg_level = DBG_HIGH;
int calib = 1;

void scan(scan_image *image, int mode);
void init_image(scan_image **image);
int main(int argc, char **argv);



static void read_calibration_data(scan_image *image);
static void read_scan_data(scan_image *image);
static void tiff_open(TIFF **tiffile, scan_image *image);
static void tiff_close(TIFF *tiffile);
static void wakeup(void);
static int power_sequence(int sequence_code);
static void wait_for_carriage(void);
static int move_carrige_forward(float distance);
int move_carrige_backward(float distance);
void parse_options(scan_image *image, int argc, char **argv);


/*-------------------------------------------------------------*/
/*
 * switches the scanner online
 */
void wakeup(void)
{
    int i;
    unsigned char data[] = {0x73, 0x72, 0x73, 0x73, 
			    0x72, 0x73, 0x73, 0x72};
    
    clear_timeout();

    outb(0x04, port + 2);  
    outb(0x70, port);
    outb(0x72, port);
    for (i = 0; i < 8; i++){
	outb(0x73, port);
	outb(0xF3, port); 
	outb(0x73, port);
	outb(0xF3, port);
	outb(0x73, port);
    }
    outb(0x7F, port);
    outb(0x04, port + 2);

    clear_timeout();

    outb(0x04, port + 2);  
    outb(0x70, port);
    outb(0x72, port);
    for (i = 0; i < 8; i++){
	outb(data[i], port);
	outb(data[i] + 0x80, port);
	outb(data[i], port);
	outb(data[i] + 0x80, port);
	outb(data[i], port);
    }
    outb(0x7F, port);
    outb(0x04, port + 2);

    DBG(DBG_LOW, "wakeup: done\n");    
}




/*-------------------------------------------------------------*/
/*  
 * This is the sequence begining with 10-18-10-.... 
 * not yet understood - but without it nothing happens
 *
 * power_sequence is DrvMcuSendRequest from the Win driver
 */


int power_sequence(int sequence_code)
{
    int i, status, timeout = 0;
    unsigned char back;
    long long timer;

    DBG(DBG_HIGH, "power_sequence:  code: %04x\n", sequence_code); 

    epp_read_one(&back, 0x0F);
    epp_write_one(0x10, 0x07); 
    epp_read_one(&back, 0x0F);
    epp_write_one(0x18, 0x07);
    epp_read_one(&back, 0x0F);
    
    for (i = 15; i >= 0 ; i--) {
	epp_write_one(0x10, 0x07);
	
	if (i!=15) {
	    init_timer(&timer, LOOP_TIMEOUT);
	    do 
		epp_read_one(&back, 0x0E); /* win driver drops line */
	    while ((back & 0x80) && !(timeout = check_timer(&timer)));  
	}	
	epp_read_one(&back, 0x0F); 
	epp_write_one(0x12 + ((sequence_code >> i) & 0x01), 0x07);
	epp_read_one(&back, 0x0F);

	init_timer(&timer, LOOP_TIMEOUT);
	do 
	    epp_read_one(&back, 0x0E);
	while ((back & 0x80) && !(timeout = check_timer(&timer)));  
	status = epp_read_one(&back, 0x0F);
    }
    epp_write_one(0x10, 0x07);

    if ((status == TIMEOUT) || (timeout == TIMEOUT))
	DBG(DBG_TIMEOUT | DBG_HIGH, 
	    "power_sequence: failed - TIMEOUT\n"); 

    return status;
}
static void wait_for_carriage(void)
{
    unsigned char back;
    long long timer;
    
    init_timer(&timer, 15000000);
    do {
        scan_delay(75000);
        epp_read_one(&back, 0x0E);
    } while (!(back & 0x40) && !(check_timer(&timer)));
}


/*-------------------------------------------------------------*/
/* 
 * opens a tiff file for writing
 */

void tiff_open(TIFF **tiffile, scan_image *image)
{
    *tiffile = TIFFOpen(image->filename, "w");
    if (!tiffile) {
	DBG(DBG_HELPERS, "Error during save: tiff_open did not open file \n");
	return;
    }
    
    TIFFSetField(*tiffile, TIFFTAG_IMAGEWIDTH, image->width);
    TIFFSetField(*tiffile, TIFFTAG_IMAGELENGTH, image->height);
    TIFFSetField(*tiffile, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
    TIFFSetField(*tiffile, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField(*tiffile, TIFFTAG_ROWSPERSTRIP,
                 TIFFDefaultStripSize(*tiffile, -1));

    switch( image->colormode)
    {
        case  RGB_MODE:
            TIFFSetField(*tiffile, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
            TIFFSetField(*tiffile, TIFFTAG_SAMPLESPERPIXEL, 3);
            TIFFSetField(*tiffile, TIFFTAG_BITSPERSAMPLE, 8);
            TIFFSetField(*tiffile, TIFFTAG_COMPRESSION, COMPRESSION_LZW &
                         image->lzw);
            break;
        case  GRAY_MODE:
            TIFFSetField(*tiffile, TIFFTAG_PHOTOMETRIC, 1);
            TIFFSetField(*tiffile, TIFFTAG_SAMPLESPERPIXEL, 1);
            TIFFSetField(*tiffile, TIFFTAG_BITSPERSAMPLE, 8);
            TIFFSetField(*tiffile, TIFFTAG_COMPRESSION, COMPRESSION_LZW &
                         image->lzw);
            break;
        case  BW_MODE:
        case  BW_MODE_RED:
        case  BW_MODE_GREEN:
        case  BW_MODE_BLUE:
            TIFFSetField(*tiffile, TIFFTAG_PHOTOMETRIC, 1);
            TIFFSetField(*tiffile, TIFFTAG_SAMPLESPERPIXEL, 1);
            TIFFSetField(*tiffile, TIFFTAG_BITSPERSAMPLE, 1);
            if ( image->lzw == 1)
                TIFFSetField(*tiffile, TIFFTAG_COMPRESSION, 1);
            else
                TIFFSetField(*tiffile, TIFFTAG_COMPRESSION, 2);
            break;
        case  RED_MODE:
        case  GREEN_MODE:
        case  BLUE_MODE:
            TIFFSetField(*tiffile, TIFFTAG_PHOTOMETRIC, 1);
            TIFFSetField(*tiffile, TIFFTAG_SAMPLESPERPIXEL, 1);
            TIFFSetField(*tiffile, TIFFTAG_BITSPERSAMPLE, 8);
            TIFFSetField(*tiffile, TIFFTAG_COMPRESSION, COMPRESSION_LZW &
                         image->lzw);
            break;
    }
    
}


/*-------------------------------------------------------------*/
/* 
 * closes a tiff file 
 */

void tiff_close(TIFF *tiffile)
{
    TIFFClose(tiffile);   
    DBG(DBG_SCAN, "tiff_close: file saved\n");
}


/*-------------------------------------------------------------*/

void read_scan_data(scan_image *image)
{
#define BUFFSIZE 0x20
    int oflag = BUFFSIZE-1;
    unsigned char* tiffdata;
    TIFF *tiffile;
    int y, i, j, color; /* 1 = red ; 2 = green ; 4 = blue */
    unsigned char pa, pb, back;
    
    
    int width,  ypixcount, num_colors, resolution, total_width;
    int pix_read_count, pix_write_count, akku;
    int pointer = 0 , offset_flag;
    int red_ptr = 0, green_ptr = 0, blue_ptr = 0, read_ptr = 0;
    int phys_line = 0;
    char line_status[BUFFSIZE];
    char *data;
    
    width  = image->width;
    ypixcount = image->height;
    num_colors = image->num_colors;
    resolution = image->resolution;
    data = (char*)malloc ( width * sizeof(char));

    for( i = 0; i < BUFFSIZE; i++)
        line_status[i] = 0;

    if ( image->colormode & 0x0010 )
    {
        total_width = image->width;
        num_colors = 1;
        offset_flag = 0;
    }
    else
    {
        total_width = 3 * image->width;
        num_colors = 3;
        offset_flag = 1;
    }
    
    
    tiffdata = malloc( (total_width) * BUFFSIZE );
    tiff_open(&tiffile, image);
    
    while(image->y_table_int[phys_line] == 0x00 )
	phys_line++;

    for (y = 0; y < ypixcount ; y++){
	while(line_status[read_ptr] != ( 0x07 & image->colormode ))
        {
	    for ( color = 1; color < 8; color = color << 1){
		do {
		    back = epp_read_one(&pa, 0x06);  
		    back = epp_read_one(&pb, 0x05);
                } while ((( i = pa * 0x100 + pb) < total_width -1 ));
//                printf("%d ",i);

                switch (color & image->y_table_int[phys_line]){
		case 1:
		    pointer = red_ptr * total_width;
		    line_status[red_ptr] =  line_status[red_ptr] | 1;
		    red_ptr = ++red_ptr & oflag;
		    break;
		case 2:
		    pointer = green_ptr * total_width + 1 * offset_flag;
		    line_status[green_ptr] =  line_status[green_ptr] | 2;
		    green_ptr = ++green_ptr & oflag;
		    break;
		case 4:
		    pointer = blue_ptr * total_width + 2 * offset_flag;
		    line_status[blue_ptr] =  line_status[blue_ptr] | 4;
		    blue_ptr = ++blue_ptr & oflag;
		    break;
		}

                
                if (0xf & image->y_table_int[phys_line]){
                    epp_read(data,width,FIFO);
                    if (color & image->y_table_int[phys_line])
                        for (i = 0; i < width ; i++){
                            tiffdata[pointer] = data[i];
                            pointer += num_colors;
                        }
                }
            }
	    phys_line++;
        }
    if ( image->colormode & 0x0100 )
    {
        pix_read_count = pix_write_count = read_ptr * total_width;
        for (i = 0; i < width ; i++)
        {
            akku = (tiffdata[pix_read_count    ] & 0xff) * 30 +
                   (tiffdata[pix_read_count + 1] & 0xff) * 59 +
                   (tiffdata[pix_read_count + 2] & 0xff) * 11 ;
            tiffdata[pix_read_count    ] = 0x00;
            tiffdata[pix_read_count + 1] = 0x00;
            tiffdata[pix_read_count + 2] = 0x00;
            tiffdata[pix_write_count] = (akku / 100) & 0xff;
            pix_read_count += num_colors;
            pix_write_count++;
        }
    }
    
    if ( image->colormode & 0x1000 )
    {
        pix_read_count = pix_write_count = read_ptr * total_width;
        for (i = 0; i < (width / 8) ; i++)
        {
            akku= 0;
            for (j = 0; j < 8 ; j++)
            {
                akku = akku << 1;
                if ((tiffdata[pix_read_count] & 0xff) > 0x7f)
                    akku = akku | 0x01;
                pix_read_count++;
            }
            tiffdata[pix_write_count] = akku  & 0xff;
            pix_write_count++;
        }
    }
    
    TIFFWriteScanline(tiffile,tiffdata + read_ptr * total_width, y , 0);
    line_status[read_ptr] = 0;
    read_ptr = ++read_ptr & oflag;
    }
    tiff_close(tiffile);           
    free(tiffdata);
}

void read_calibration_data(scan_image *image)
{
    unsigned char back, pa, pb;
    unsigned int pointer = 0, y, i;
    unsigned int height = image->height;
    unsigned int total_width = image->num_colors * image->width;

    for (y = 0; y < height ; y++){
        do {
            back = epp_read_one(&pa, 0x06);
            back = epp_read_one(&pb, 0x05);
        } while ((( i = pa * 0x100 + pb) < total_width -1 ));
/*          printf("%d ",i); */
        epp_read((image->data) + pointer ,total_width,FIFO);
        pointer += total_width;
        
    }
}
/*-------------------------------------------------------------*/
/*
 * moves the lamp forward. distance in inch
 *
 * same as DrvMoveCarrigeForward  
 * theory: 
 *    - power 0x2000
 *    - out 5    - 0x00, 0x0D, 0x0D, 0x0D
 *    - in  0x0F
 *    - out 5    - 0x00, 0x0D, 0x0D, 0x0D
 *    - out 6    - 0x24, 0x20, 0x24, 0xBC
 *    - power 0x2000
 *    - out 5    - 0x00, 0x0D, 0x0D, 0x0D
 *    - out 5    - 0x00, 0x00, 0x00, 0x00
 *    - out 6    - 0x24, 0x20, 0x24, 0x3C
 *    - out 5    - 0x00, 0x00, 0x00, 0x00
 *    - out 6    - 0x24, 0x20, 0x24, 0x3C
 *    - power 0x2000
 *    - out 5    - 0x00, 0x00, 0x00, 0x00
 *    - out 5    - 0x0D, 0x0D, 0x0D, 0x0D
 *    - out 6    - 0x24, 0x20, 0x24, 0x3C
 *    - out 6    - 0x24, 0x20, 0x24, 0x3C
 *    - out 6    - 0x26, 
 *    - out 6    - 0x24, 
 *    [- in  0x0E          
 *     - in  0x0E    - repeat !0x80
 *     - out 6   - 0x24
 *     - out 6   - 0x26
 *     -]out 6   - 0x24  - reapeat [..]
 *    -  out 5   - 0x0D, 0x0D, 0x0D, 0x0D
 *    - power 0x40B0
 *    - power 0x8000
 *
 * values are different for 150 dpi !!!
 */
int move_carrige_forward(float distance) 
{
    int i,status, timeout = 0;
    long long timer;
    unsigned char back;
#if USE_SCHEDULER
    struct sched_param priority;
#endif

    /* change this if you know the meaning*/
    unsigned char value_1 = 0x00, value_2 = 0x24;

    power_sequence(S_2000);

    epp_write_one(value_1, 0x05);
    status =  epp_read_one(&back, 0x05);
    epp_write_one(value_1, 0x05);
    epp_write_one(value_2, 0x06);

    power_sequence(S_2000);

    epp_write_one(value_1, 0x05);
    epp_write_one(0x00, 0x05);
    epp_write_one(value_2 & 0x7F, 0x06);
    epp_write_one(0x00, 0x00);
    epp_write_one(value_2 & 0x7F, 0x06);
    
    power_sequence(S_2000);

    epp_write_one(0x00, 0x05);
    epp_write_one(0x05, 0x05);                 /* Speed ! */ 
    epp_write_one(value_1, 0x06);
    epp_write_one(value_1, 0x06);
    epp_write_one(0x26, 0x06);    
    epp_write_one(0x24, 0x06);

    /* we need this to make a smooth move */
#if USE_SCHEDULER
    priority.sched_priority = 1;
    sched_setscheduler(0, SCHED_FIFO, &priority);
#endif

    for (i = 0; i < distance * Y_RESOLUTION; i++) {
	status = epp_read_one(&back, 0x0E);
	init_timer(&timer, LOOP_TIMEOUT);
	do {
	    status = epp_read_one(&back, 0x0E);  
	} while ((back & 0x80) && !(timeout = check_timer(&timer))); /* ready for next move? */	
	if (timeout)
	    DBG(DBG_TIMEOUT | DBG_MEDIUM, "move_carrige_forward:  timeout!\n");
	epp_write_one(0x24, 0x06);
	epp_write_one(0x26, 0x06);    /* move */ /*bit 15 is direction bit */
	epp_write_one(0x24, 0x06);
    }
    epp_write_one(0x0D, 0x05);
#if USE_SCHEDULER
    priority.sched_priority = 0;
    sched_setscheduler(0, SCHED_OTHER, &priority);
#endif
    power_sequence(S_40B0);
    power_sequence(S_8000);
    return status;
}


/*-------------------------------------------------------------*/




/*-------------------------------------------------------------*/
/*
 * moves the lamp backward. distance in inch
 *
 * it is not a windows trace, it's just
 * move_carrige_forward with other values
 */

int move_carrige_backward(float distance) 
{  
    int i, status, timeout = NO_TIMEOUT;
    long long timer;
    unsigned char back;
#if USE_SCHEDULER
    struct sched_param priority;
#endif

    /* change this if you know the meaning*/
    unsigned char value_1 = 0x00, value_2 = 0x24;
    
    power_sequence(S_2000);
    
    epp_write_one(value_1, 0x05);
    status =  epp_read_one(&back, 0x05);
    epp_write_one(value_1, 0x05);
    epp_write_one(value_2, 0x06);
    
    power_sequence(S_2000);
    
    epp_write_one(value_1, 0x05);
    epp_write_one(0x00, 0x05);
    epp_write_one(value_2 & 0x7F, 0x06); /* 0x7F just for fun */
    epp_write_one(0x00, 0x00);
    epp_write_one(value_2 & 0x7F, 0x06);
    
    power_sequence(S_2000);
    
    epp_write_one(0x00, 0x05);
    epp_write_one(0x00, 0x05);                 /* Speed ? */ 
    epp_write_one(value_1, 0x06);
    epp_write_one(value_1, 0x06);
    epp_write_one(0x22, 0x06);    
    epp_write_one(0x20, 0x06);
    /* we need this to make a smooth move */
#if USE_SCHEDULER
    priority.sched_priority = 1;
    sched_setscheduler(0, SCHED_FIFO, &priority);
#endif
    for (i = 0; i < distance * Y_RESOLUTION; i++) {
	status = epp_read_one(&back, 0x0E);
	init_timer(&timer, LOOP_TIMEOUT);
	do {
	    status = epp_read_one(&back, 0x0E);  
	} while ((back & 0x80) && !(timeout = check_timer(&timer))); /* ready for next move? */
	if (timeout)
	    DBG(DBG_TIMEOUT | DBG_MEDIUM, "move_carrige_backward:  timeout!\n");
	epp_write_one(0x20, 0x06);
	epp_write_one(0x22, 0x06);    /* move */ /*bit 15 is direction bit */
	epp_write_one(0x20, 0x06);
    }
    epp_write_one(0x0D, 0x05);
#if USE_SCHEDULER
    priority.sched_priority = 0;
    sched_setscheduler(0, SCHED_OTHER, &priority);
#endif
    power_sequence(S_40B0);
    power_sequence(S_8000);
    return status;
}




/*-------------------------------------------------------------*/
/*
 *  ....
 */
void scan(scan_image *image, int mode)
{
    int i;
//    unsigned char back;
    
    calc_y_table(image);
    calc_x_table(image);
    calc_color_table(image);
    
    /* skip calibration area */
    if (mode == SCAN)
	move_carrige_forward( CALIBRATION_ARRAY_SIZE );
    else
        move_carrige_forward(image->from_top);
    
/*      move_carrige_forward(0); */
/*      move_carrige_forward(image->from_top); */

    if (calib)
    {
        write_LM9811( image->LM9811_reg_PGA , LM9811_PGA );
        write_LM9811( image->LM9811_reg_VGA , LM9811_VGA );
        write_LM9811( image->LM9811_reg_DAC , LM9811_DAC );
        epp_write(image->LM9811_arr_PGA, LM9811_PGA_array_size,
                  CALIBRATION_TABLE);
    }
    

         /*Reset FIFO*/
    epp_write_one(0x64, 0x06);
    epp_write_one(0x24, 0x06);  


    for(i = 0; i < 3; i++)
        epp_write((image->color_table)[i], color_table_size, COLOR_TABLE); 
    
    epp_write(image->x_table, x_table_size, X_TABLE); 
    epp_write(image->y_table, y_table_size, Y_TABLE);
    
    epp_write_one(0x40 | (image->speed & 0xf), 0x05);
                               /* The bits 0..3 are for brightness,
                                  integration-time of ccd */
    
//    epp_write_one(0xBC, 0x06); /* bit 5 = lamp on */
    epp_write_one(0xbD, 0x06); /* bit 0 = start scan, bit 7 = must be,
                        	  bit 6 = must not be, bit 5 = lamp on,
                                  bit 4 has to do something with calib. */

    if (mode == SCAN)
        read_scan_data(image);
    else
        read_calibration_data(image);

    power_sequence(CARRIGE_RETURN);
    wait_for_carriage();
 
}

/*-------------------------------------------------------------*/
void init_image(scan_image **image) 
{
    int i;

    *image = malloc (sizeof(scan_image));

    (*image)->resolution = 50;
    (*image)->width = (*image)->resolution * MAX_WIDTH;
    (*image)->height = (*image)->resolution * MAX_HEIGHT ;
    (*image)->speed = 5;
    (*image)->from_top = 0;
    (*image)->from_left = 0;
    (*image)->num_colors = 3;
    (*image)->colormode = RGB_MODE;

    for (i = 0; i < 3; i++) {
	((*image)->color)[i].gamma = 1;
	((*image)->color)[i].contrast = 0;
	((*image)->color)[i].brightness = 0;
	((*image)->color_table)[i] = malloc(color_table_size);
    }

    (*image)->LM9811_arr_PGA = malloc(LM9811_PGA_array_size);
    (*image)->LM9811_reg_PGA = 0;
    (*image)->LM9811_reg_VGA = 0;
    (*image)->LM9811_reg_DAC = 0;
    (*image)->y_table =(char*) malloc(y_table_size);
    (*image)->y_table_int =(char*) malloc(y_table_size);
    (*image)->x_table =(char*) malloc(x_table_size);

    (*image)->data = NULL;

    (*image)->filename = malloc(100);
    (*image)->filename = "primax_scan.tif";
    (*image)->lzw = 0xffff;
}

/*-------------------------------------------------------------*/
/* 
 * parse command line input
 */
void parse_options(scan_image *image, int argc, char **argv)
{
    int i, index = 0;
    int c, switch_lamp = 0xFF;
    char *tmp;
    unsigned char back, status;
    double width = -1, height = -1;
    double temp;


    static struct option options[] = {
	{"help"       ,0, 0, 'h'},
	{"verbose"    ,2, 0, 'v'},
	{"port"       ,1, 0, 'P'},
	{"dim"        ,1, 0, 'd'},
	{"res"        ,1, 0, 'r'},
	{"pos"        ,1, 0, 'p'},
	{"version"    ,0, 0, 'V'},
	{"contrast"   ,1, 0, 'c'},
	{"brightness" ,1, 0, 'b'},
	{"gamma"      ,1, 0, 'g'},
	{"file"       ,1, 0, 'f'},
	{"lamp"       ,1, 0, 'l'},
	{"RED"        ,0, 0, 'D'},
	{"GREEN"      ,0, 0, 'N'},
	{"BLUE"       ,0, 0, 'E'},
	{"RGB"        ,0, 0, 'R'},
	{"GRAY"       ,0, 0, 'G'},
	{"BW"         ,0, 0, 'B'},
	{"speed"      ,1, 0, 's'},
	{"Scanner"    ,1, 0, 'S'},
	{"Compression",1, 0, 'C'},
	{"NoCalib    ",0, 0, 'o'},
	{0            ,0, 0,   0}
    };
    
    dbg_level = DBG_HELPERS;
    DBG(DBG_HELPERS,"parse_options:\n");
    while (1) {
	 c = getopt_long(argc, argv, "hv:P:d:Vp:d:r:c:b:g:f:l:DNERGBs:S:C:o",
			 options, &index);
	 if (c == -1)
	     break;
	 switch (c){
         case 'o':
             calib = 0;  
	     break;
         case 'P':
	     port = strtol (optarg, (char **)NULL, 0x10);
	     DBG(DBG_HELPERS, "  port:               0x%03x\n", port);
	     break;
	 case 'S':
	     RESOLUTION = strtol (optarg, (char **)NULL, 0x00);
	     break;
	 case 'C':
	     if (strtol (optarg, (char **)NULL, 0x00) == 0)
                 image->lzw = 1;
	     break;
	 case 'v':
	     if (!optarg)
		 dbg_level = DBG_HIGH;
	     else
		 dbg_level = strtol (optarg, (char **)NULL, 0x10);
	     DBG(DBG_HELPERS, "  debug level:        0x%02x\n",
                 (unsigned char)dbg_level);
	     break;
	 case 'V':
	     printf("primax_scan version ");
	     printf(VERSION);
	     printf("\n");
	     printf("$Id: primax_scan.c,v 1.2 2000/10/24 19:12:52 chrordig Exp $"); 
	     printf("\n");
	     exit(0);
	 case 'p':
	     image->from_left = strtod(optarg, &tmp);
	     image->from_top = strtod(tmp + 1, (char **)NULL);
	     DBG(DBG_HELPERS, "  position:    X= %1.1f x Y= %1.1f inch\n", 
		 image->from_left, image->from_top);
	     break;
	 case 'd':
	     width = strtod(optarg, &tmp);
	     height = strtod(tmp + 1, (char **)NULL);
	     DBG(DBG_HELPERS, "  dimension:   X= %1.1f x Y= %1.1f inch\n", 
		 width, height);
	     break;
	 case 'r':
	     image->resolution = strtol(optarg, (char **) NULL, 10);
	     DBG(DBG_HELPERS, "  resolution:           %i dpi\n", 
		 image->resolution);
	     break;
	 case 'c':
             temp = strtod(optarg, &tmp);
             for (i = 0; i < 3; i++)
             {
                 image->color[i].contrast = temp;
             }
	     break;
	 case 'b':
	     temp = strtod(optarg, &tmp);
             for (i = 0; i < 3; i++)
             {
                image->color[i].brightness = temp;
             }
	     break;
	 case 'g':
	     temp = strtod(optarg, &tmp);
             for (i = 0; i < 3; i++)
             {
                 image->color[i].gamma = temp;
             }
	     break;
	 case 'f':
	     image->filename = optarg;
	     DBG(DBG_HELPERS, "  filename:     %s\n",image->filename); 
	     break;
	 case 'D':
	     image->colormode = (image->colormode & 0xF0F1) | 0x0011;
	     break;
	 case 'E':
	     image->colormode = (image->colormode & 0xF0F4) | 0x0014;
	     break;
	 case 'N':
	     image->colormode = (image->colormode & 0xF0F2) | 0x0012;
	     break;
	 case 'R':
	     image->colormode = (image->colormode & 0xFF0F) | 0x0007;
	     break;
	 case 'G':
	     image->colormode = (image->colormode & 0xFF0F) | 0x0107;
	     break;
	 case 'B':
             if ( image->colormode & 0x0010 )
                 image->colormode = (image->colormode & 0xF0FF) | 0x1000;
             else
                 image->colormode = BW_MODE; 
	     break;
	 case 's':
	     image->speed = (int) strtol (optarg, (char **)NULL, 0) & 0xf;
	     break;
	 case 'l':
	     if (strcmp((const char *)optarg, "off"))
		 switch_lamp = 0x20;
	     else
		 switch_lamp = 0x00;
	     DBG(DBG_HELPERS, "  lamp:                 %s\n",optarg); 
	     break;
	 case 'h':
	     printf("\n\nUsage:\n");
	     printf("-h         ,   --help            shows this message\n");
	     printf("-v [LEVEL] ,   --verbose=[LEVEL] increase debugging "
                    "output\n");
	     printf("-P PORT    ,   --port=PORT       Hex-address of the "
                    "parallel-PORT (%X)\n", port);
	     printf("-s SPEED   ,   --speed=SPEED     SPEED of the carriage "
                    "{0,1,2...15}\n");
	     printf("-S RES     ,   --Scanner=RES     max. optical RESolution "
                    "of the Scanner\n"
                    "                                 {300,600} dpi\n");
	     printf("-V         ,   --version         display version\n");
	     printf("-p LxT     ,   --pos=LxT         position form Left "
                    "x Top in inch\n");
	     printf("-d WxH     ,   --dim=WxH         Width x Heigth of "
                    "scan area in\n"
                    "                                 inch (max. %2.2f"
                    " x %2.2f)\n", MAX_WIDTH, MAX_HEIGHT);
	     printf("-c C       ,   --contrast=C      Contrast "
                    "{(-2000)..(1000)}\n");
	     printf("-b B       ,   --brightness=B    Brightness "
                    "{(-200)..(200)}\n");
	     printf("-g G       ,   --gamma=G         Gamma {(0.2)..(5)}\n");
	     printf("-r RES     ,   --res=RES         RESolution in dpi (max. "
                    "%d) see option -S\n", RESOLUTION);
	     printf("-f NAME    ,   --file=NAME       NAME for tiff-file\n");
	     printf("-C val     ,   --Compression=val val=0 switches "
                    "compression off\n");
	     printf("-l on/off  ,   --lamp=on/off     turn lamp on/off\n");
             printf("\n");
             printf("-D         ,   --RED             only RED-channel\n");
             printf("-E         ,   --BLUE            only BLUE-channel\n");
             printf("-N         ,   --GREEN           only GREEN-channel\n");
             printf("-R         ,   --RGB             24-Bit RGB-mode "
                    "(default)\n");
             printf("-G         ,   --GRAY            8-Bit GRAY-mode\n");
             printf("-B         ,   --BW              1-Bit Black-White-"
                    "mode (possible with\n"
                    "                                 all other modes)\n");
	     printf("\n\n\n");
	     exit(0);
	 }
    }
    if ((width != -1) || (height != -1))
        {
            image->width = (int)(image->resolution * width);
            image->height = (int)(image->resolution * height);
        }
    
    DBG(DBG_HELPERS, "  dimension:        %3ix%3i pixel\n", 
	image->width, image->height);

    /* switch lamp and try to detect scanner */
    if (switch_lamp != 0xFF) {
	DBG(DBG_HELPERS, "parse_options:       switching lamp\n");
	epp_on();
	wakeup();
	epp_write_one(switch_lamp, 0x06);
	status = epp_read_one(&back, 0x0E);
	if (!status) {
            switch (back & 0x0F) {
                case 0x02:
                    printf("\nLooks like a Primax Colorado Direct\n");
                    break;
                case 0x05:
                    printf("\nLooks like a Primax Colorado Direct 600\n");
                    break;
                case 0x44:
                    printf("\nLooks like a Primax Colorado D600_OEM\n");
                    break;
                default:
                    printf("\nCan not identify this scanner. code: %02x\n", back);
	    }
	}
	else {
	    printf("\n:-(\nScanner did not switch to EPP mode");
	    printf(".....not compatible?\n");
	}
	power_sequence(GO_OFFLINE);
	epp_off();
	exit(0);
    }
}

/*-------------------------------------------------------------*/

int main(int argc, char **argv)
{
    scan_image *image;
    char i,j;
    

    init_image( &image);
    parse_options(image, argc, argv);
//  /dev/lpx must be allocated!
    epp_on();
    wakeup();
/*           printf("\n"); */
   
    power_sequence(CARRIGE_RETURN);

/*          for (j = 0; j < 0x10; j++) */
/*          { */
/*              epp_read_one(&i,j); */
/*              printf("%x,%2x|",j,i&0xff ); */
/*          } */
/*          printf("\n"); */
/* epp_write_one(0,3);*/
    if (calib)
        calibrate_LM9811(image);
    power_sequence(CARRIGE_RETURN);
/*      power_sequence(S_4000); */
/*          for (j = 0; j < 0x10; j++) */
/*          { */
/*              epp_read_one(&i,j); */
/*              printf("%x,%2x|",j,i&0xff ); */
/*          } */
/*          printf("\n"); */
    
    scan(image, SCAN);
/*    power_sequence(CARRIGE_RETURN);*/
/*          for (j = 0; j < 0x10; j++) */
/*          { */
/*              epp_read_one(&i,j); */
/*              printf("%x,%2x|",j,i&0xff ); */
/*          } */
/*          printf("\n"); */
    power_sequence(CARRIGE_RETURN);
    wait_for_carriage();
    power_sequence(GO_OFFLINE);
     epp_off();
   free(image);
    return 0;
}  




/* This stuff is for emacs
 * Local variables:
 * c-basic-offset: 4
 * End:       
 */










