from typing import Type

import attr
import pytest

from cattrs import Converter, GenConverter


@pytest.mark.parametrize("converter_cls", [GenConverter, Converter])
def test_inheritance(converter_cls):
    @attr.define
    class A:
        i: int

    @attr.define
    class B(A):
        j: int

    converter = converter_cls()

    assert A(1) == converter.structure(dict(i=1), A)
    assert B(1, 2) == converter.structure(dict(i=1, j=2), B)


@pytest.mark.parametrize("converter_cls", [Converter, GenConverter])
def test_gen_hook_priority(converter_cls: Type[Converter]):
    """Autogenerated hooks should not take priority over manual hooks."""
    converter = converter_cls()

    @attr.define
    class A:
        i: int

    @attr.define
    class B(A):
        pass

    # This will generate a hook.
    assert converter.structure({"i": 1}, B) == B(1)

    # Now we register a manual hook for the superclass.
    converter.register_structure_hook(A, lambda o, T: T(o["i"] + 1))

    # This should still work, but using the new hook instead.
    assert converter.structure({"i": 1}, B) == B(2)
