/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtLanguage.h"

#include <model/config/LanguageList.h>
#include <model/config/ConfigManager.h>
#include <model/config/Config.h>

#include <util/File.h>
#include <util/Logger.h>

#include <settings/Settings.h>
#include <settings/SettingsXMLSerializer.h>

#include <qtutil/KeywordTranslator.h>

#include <QtCore/QCoreApplication>
#include <QtCore/QDir>
#include <QtCore/QLocale>
#include <QtCore/QTranslator>

static const char * LANGUAGE_FILE_EXTENSION = "qm";
static const char * LANG_DIR = "lang";

static const char * BRANDING_FILENAME = "config/branding.xml";

static KeywordTranslator::KeywordHash readKeywordHash(const std::string& fileName) {
	FileReader reader(fileName);
	KeywordTranslator::KeywordHash hash;
	if (!reader.open()) {
		LOG_ERROR("Couldn't open file " + fileName);
		return hash;
	}

	std::string data = reader.read();

	Settings settings;
	SettingsXMLSerializer serializer(settings);
	serializer.unserialize(data);

	StringList list = settings.getAllKeys();
	StringList::const_iterator
		it = list.begin(),
		end = list.end();
	for (; it!=end; ++it) {
		std::string keyword = *it;
		std::string value = settings.getStringKeyValue(keyword);
		hash[QString::fromUtf8(keyword.c_str())] = QString::fromUtf8(value.c_str());
	}

	return hash;
}

QtLanguage::QtLanguage(QObject * parent)
	: QObjectThreadSafe(parent) {

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	config.valueChangedEvent += boost::bind(&QtLanguage::configChangedEventHandler, this, _1);

	_qtTranslator = new QTranslator(QCoreApplication::instance());

	KeywordTranslator::KeywordHash hash = readKeywordHash(config.getResourcesDir() + "/" + BRANDING_FILENAME);

	_appTranslator = new KeywordTranslator(QCoreApplication::instance());
	_appTranslator->setKeywordHash(hash);
	_appTranslator->setFallbackTranslator(_qtTranslator);
	QCoreApplication::installTranslator(_appTranslator);

	loadLanguageFromConfig();
}

QtLanguage::~QtLanguage() {
}

void QtLanguage::configChangedEventHandler(const std::string & key) {
	typedef PostEvent1<void (const std::string &), std::string> MyPostEvent;
	MyPostEvent * event = new MyPostEvent(boost::bind(&QtLanguage::configChangedEventHandlerThreadSafe, this, _1), key);
	postEvent(event);
}

void QtLanguage::configChangedEventHandlerThreadSafe(const std::string & key) {
	if (key == Config::LANGUAGE_KEY) {
		loadLanguageFromConfig();
	}
}

QString QtLanguage::getLocaleFileName() {
	QString localeName;
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	if (config.getLanguage() == Config::LANGUAGE_AUTODETECT_KEYVALUE) {
		localeName = QLocale::system().name();
		if (localeName == "C") {
			// No locale defined, just get out
			return QString::null;
		}
	} else {
		localeName = QString::fromUtf8(config.getLanguage().c_str());
	}
	return localeName;
}

QString QtLanguage::getQtLocaleFileName() {
	QString localeName = QtLanguage::getLocaleFileName();
	if (!localeName.isEmpty()) {
		return "qt_" + localeName;
	} else {
		return QString::null;
	}
}

QString QtLanguage::getQuteComLocaleFileName() {
	QString localeName = QtLanguage::getLocaleFileName();
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString langDir = QString::fromUtf8(config.getResourcesDir().c_str()) + LANG_DIR;
	if (!localeName.isEmpty()) {
		return langDir + QDir::separator() + "qtqutecom_" + localeName;
	} else {
		return QString::null;
	}
}

void QtLanguage::loadLanguageFromConfig() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	QString qtTranslationName = getQtLocaleFileName();

	QString langDir = QString::fromUtf8(config.getResourcesDir().c_str()) + LANG_DIR;
	bool qtTranslationLoaded = _qtTranslator->load(qtTranslationName, langDir);

#ifdef QT_TRANSLATION_DIR
	if (!qtTranslationLoaded) {
		qtTranslationLoaded = _qtTranslator->load(qtTranslationName, QT_TRANSLATION_DIR);
	}
#endif
	if (!qtTranslationLoaded) {
		LOG_INFO("no Qt translation available for locale '" + getLocaleFileName().toUtf8() + "'");
	}

	LOG_DEBUG(getQuteComLocaleFileName().toUtf8());
	if (!_appTranslator->load(getQuteComLocaleFileName())) {
		LOG_INFO("no application translation available for locale '" + getLocaleFileName().toUtf8() + "'");
	}
}

static QString getIso639CodeFromFileName(const QString & fileName) {
	// qtqutecom_en.qm -> en
	QRegExp rx("qtqutecom_([^.]+)");
	rx.indexIn(fileName);
	QStringList list = rx.capturedTexts();
	if (list.size() != 2) {
		return QString();
    }
    QString code = list[1];
    //code.replace("_", "-");
    return code/*.toLower()*/;
}

QStringList QtLanguage::getAvailableLanguages() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	QStringList languageList;
	QDir dir(QString::fromUtf8(config.getResourcesDir().c_str()) + LANG_DIR);

	Q_FOREACH(QFileInfo fileInfo, dir.entryInfoList()) {

		if (fileInfo.isFile() &&
			fileInfo.size() &&
			(fileInfo.completeSuffix().toUtf8() == LANGUAGE_FILE_EXTENSION)) {

			QString iso639Code = getIso639CodeFromFileName(fileInfo.fileName());
			if (iso639Code.isEmpty()) {
				// This can happen if the translation dir contains other files
				// (like Qt translations)
				continue;
			}
			std::string language = LanguageList::getLanguageName(std::string(iso639Code.toUtf8()));
			languageList += QString::fromUtf8(language.c_str());
		}
	}

	return languageList;
}
