% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arg.R
\name{missing_arg}
\alias{missing_arg}
\alias{is_missing}
\alias{maybe_missing}
\title{Generate or handle a missing argument}
\usage{
missing_arg()

is_missing(x)

maybe_missing(x, default = missing_arg())
}
\arguments{
\item{x}{An object that might be the missing argument.}

\item{default}{The object to return if the input is missing,
defaults to \code{missing_arg()}.}
}
\description{
These functions help using the missing argument as a regular R
object.
\itemize{
\item \code{missing_arg()} generates a missing argument.
\item \code{is_missing()} is like \code{\link[base:missing]{base::missing()}} but also supports
testing for missing arguments contained in other objects like
lists. It is also more consistent with default arguments which
are never treated as missing (see section below).
\item \code{maybe_missing()} is useful to pass down an input that might be
missing to another function, potentially substituting by a
default value. It avoids triggering an "argument is missing" error.
}
}
\section{Other ways to reify the missing argument}{

\itemize{
\item \code{base::quote(expr = )} is the canonical way to create a missing
argument object.
\item \code{expr()} called without argument creates a missing argument.
\item \code{quo()} called without argument creates an empty quosure, i.e. a
quosure containing the missing argument object.
}
}

\section{\code{is_missing()} and default arguments}{


The base function \code{\link[=missing]{missing()}} makes a distinction between default
values supplied explicitly and default values generated through a
missing argument:\if{html}{\out{<div class="sourceCode r">}}\preformatted{fn <- function(x = 1) base::missing(x)

fn()
}\if{html}{\out{</div>}}\preformatted{## [1] TRUE
}\if{html}{\out{<div class="sourceCode r">}}\preformatted{fn(1)
}\if{html}{\out{</div>}}\preformatted{## [1] FALSE
}

This only happens within a function. If the default value has been
generated in a calling function, it is never treated as missing:\if{html}{\out{<div class="sourceCode r">}}\preformatted{caller <- function(x = 1) fn(x)
caller()
}\if{html}{\out{</div>}}\preformatted{## [1] FALSE
}

\code{rlang::is_missing()} simplifies these rules by never treating
default arguments as missing, even in internal contexts:\if{html}{\out{<div class="sourceCode r">}}\preformatted{fn <- function(x = 1) rlang::is_missing(x)

fn()
}\if{html}{\out{</div>}}\preformatted{## [1] FALSE
}\if{html}{\out{<div class="sourceCode r">}}\preformatted{fn(1)
}\if{html}{\out{</div>}}\preformatted{## [1] FALSE
}

This is a little less flexible because you can't specialise
behaviour based on implicitly supplied default values. However,
this makes the behaviour of \code{is_missing()} and functions using it
simpler to understand.
}

\section{Fragility of the missing argument object}{


The missing argument is an object that triggers an error if and
only if it is the result of evaluating a symbol. No error is
produced when a function call evaluates to the missing argument
object. For instance, it is possible to bind the missing argument
to a variable with an expression like \code{x[[1]] <- missing_arg()}.
Likewise, \code{x[[1]]} is safe to use as argument, e.g. \code{list(x[[1]])}
even when the result is the missing object.

However, as soon as the missing argument is passed down between
functions through a bare variable, it is likely to cause a missing
argument error:\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- missing_arg()
list(x)
#> Error:
#> ! argument "x" is missing, with no default
}\if{html}{\out{</div>}}

To work around this, \code{is_missing()} and \code{maybe_missing(x)} use a
bit of magic to determine if the input is the missing argument
without triggering a missing error.\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- missing_arg()
list(maybe_missing(x))
#> [[1]]
#>
}\if{html}{\out{</div>}}

\code{maybe_missing()} is particularly useful for prototyping
meta-programming algorithms in R. The missing argument is a likely
input when computing on the language because it is a standard
object in formals lists. While C functions are always allowed to
return the missing argument and pass it to other C functions, this
is not the case on the R side. If you're implementing your
meta-programming algorithm in R, use \code{maybe_missing()} when an
input might be the missing argument object.
}

\examples{
# The missing argument usually arises inside a function when the
# user omits an argument that does not have a default:
fn <- function(x) is_missing(x)
fn()

# Creating a missing argument can also be useful to generate calls
args <- list(1, missing_arg(), 3, missing_arg())
quo(fn(!!! args))

# Other ways to create that object include:
quote(expr = )
expr()

# It is perfectly valid to generate and assign the missing
# argument in a list.
x <- missing_arg()
l <- list(missing_arg())

# Just don't evaluate a symbol that contains the empty argument.
# Evaluating the object `x` that we created above would trigger an
# error.
# x  # Not run

# On the other hand accessing a missing argument contained in a
# list does not trigger an error because subsetting is a function
# call:
l[[1]]
is.null(l[[1]])

# In case you really need to access a symbol that might contain the
# empty argument object, use maybe_missing():
maybe_missing(x)
is.null(maybe_missing(x))
is_missing(maybe_missing(x))


# Note that base::missing() only works on symbols and does not
# support complex expressions. For this reason the following lines
# would throw an error:

#> missing(missing_arg())
#> missing(l[[1]])

# while is_missing() will work as expected:
is_missing(missing_arg())
is_missing(l[[1]])
}
