# ifndef _SKIT_OUTPS_H
# define _SKIT_OUTPS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// OUT PS: Matrix Output 
//        in Postscript format
// 
// author: Pierre.Saramito@imag.fr
//
// date: 21 january 1997
//
// note: inspired from SPARSKIT2/INOUT/inout.f
//       PSPLTM - PostScript PLoTer of a (sparse) Matrix
//       This version by loris renggli (renggli@masg1.epfl.ch), Dec 1991
//       and Youcef Saad 
//
//       extended to grayscale and color by P. Saramito
//
# include "rheolef/outps.h"
# include "rheolef/color.h"
# include "rheolef/array.h"    /* for min() max() on array */
# include "rheolef/iorheo.h"
# include "rheolef/csr.h"
namespace rheolef { 

// ps_pref with default output choices
struct ps_preferences {

	inline void reset () {


		// color scale
		color = false;

		// gray scale
		gray = false;
	}
	// cstor
	ps_preferences() { reset(); }
	
	bool color;
	bool gray;
};
// global var
extern ps_preferences ps_pref;


template <class T>
struct log10_abs_or_zero {
    T operator()(T x) { return ((x == T(0.)) ? T(0.) : log10(xabs(x))); }
};

template <
	class Size, 
	class IteratorValue,
	class IteratorIndex>
void
print_postscript (
	std::ostream& s, 
	Size nrow, 
        Size ncol, 
        Size nnz,
	IteratorValue a, 
	IteratorIndex ja, 
	IteratorIndex ia) 
{
       typedef typename std::iterator_traits<IteratorValue>::value_type  T;

    // scales
        bool logscale = iorheo::getlogscale(s);
        bool color    = iorheo::getcolor(s);

	// title of the drawing
	// const char title [256] = "Sparse Matrix";
	const char title [256] = "";

	// ptitle = position of title; 0 under the drawing, else above
	const  int ptitle = 0;
	
	// size   = size of the drawing (unit =cm) 
	const T size = 15;
	
	// units cm do dot conversion factor and a4 paper size
      	const T paperx = 21.0;
      	const T conv   = 2.54; // cm-per-inch
      	T u2dot        = 72.0/conv;

      	int maxdim = max(nrow, ncol);
      	int m = 1 + maxdim;

	// left and right margins (drawing is centered)
	T lrmrgn = (paperx-size)/2.0;

	// bottom margin : 2 cm
	T botmrgn = 2;

	// scaling factor
      	T scfct = size*u2dot/m;

	// matrix frame line witdh
        const T frlw = 0.25;

	// font size for title (cm)
        const T fnstit = 0.5;
        int    ltit = strlen(title);

	// position of title : centered horizontally
	//                     at 1.0 cm vertically over the drawing
        const T ytitof = 1.0;
        T xtit = paperx/2.0;

        T ytit = botmrgn + size*int(nrow)/T(m) + T(ytitof);

	// almost exact bounding box
        T xl = lrmrgn*u2dot - scfct*frlw/2;
        T xr = (lrmrgn+size)*u2dot + scfct*frlw/2;
        T yb = botmrgn*u2dot - scfct*frlw/2;
        
	T yt = (botmrgn+size*int(nrow)/T(m))*u2dot + scfct*frlw/2.;
        
	if (ltit == 0)
            yt = yt + (ytitof+fnstit*0.70)*u2dot;

	// add some room to bounding box
        const T delt = 10.0;
        xl -= delt;
        xr += delt;
        yb -= delt;
        yt += delt;

	// correction for title under the drawing
        if (ptitle == 0 && ltit > 0) {
            ytit = botmrgn + fnstit*0.3;
            botmrgn = botmrgn + ytitof + fnstit*0.7;
	}
   
   // print header

	s << "%!PS-Adobe-2.0\n";
	s << "%%Creator: sparskit++ 1.0, 1997, Computer Modern Artists\n";
	s << "%%Title: sparskit++ CSR matrix\n";
	s << "%%BoundingBox: " << xl << " " << yb << " " << xr << " " << yt << std::endl;
	s << "%%EndComments\n";
	s << "/cm {72 mul 2.54 div} def\n";
	s << "/mc {72 div 2.54 mul} def\n";
	s << "/pnum { 72 div 2.54 mul 20 string\n";
	s << "cvs print ( ) print} def\n";
	s << "/Cshow {dup stringwidth pop -2 div 0 rmoveto show} def\n";

	// we leave margins etc. in cm so it is easy to modify them if
	// needed by editing the output file
	s << "gsave\n";
        if (ltit > 0)
            s << "/Helvetica findfont " << fnstit << " cm scalefont setfont \n";
        s << xtit << " cm " << ytit << " cm moveto\n";
        s << "(" << title << ") Cshow\n";

        s << lrmrgn << " cm " << botmrgn << " cm translate\n";
        s << size << " cm " << m << " div dup scale\n";

	// draw a frame around the matrix
	s << frlw << " setlinewidth\n";
	s << "newpath\n";
	s << "0 0 moveto\n";
        s <<  ncol+1 << " " <<      0 << " lineto\n";
        s <<  ncol+1 << " " << nrow+1 << " lineto\n";
        s <<       0 << " " << nrow+1 << " lineto\n";
	s << "closepath stroke\n";

	s << "1 1 translate\n";
	s << "0.8 setlinewidth\n";
	s << "/p {moveto 0 -.40 rmoveto \n";
	s << "           0  .80 rlineto stroke} def\n";

	if (color) {

	    // the number of used colors
	    const int ncolor = 100;

	    T max_;
	    T min_;
	    T inf = std::numeric_limits<T>::max();
	    log10_abs_or_zero<T> filter;

	    if (logscale) {
		max_ = select_op_value (a, a+nnz, -inf, std::greater<T>(), filter);
		min_ = select_op_value (a, a+nnz,  inf, std::less<T>(),    filter);
	    } else {
	        max_ = select_max(a, a+nnz, -inf);
	        min_ = select_min(a, a+nnz,  inf);
	    }
	    palette tab = palette(min_, max_, ncolor);
	    tab.postscript_print_color_def (s);
	    
	    // color: plotting loop
            for (Index i = 0; i < nrow; i++)
                for (Index k = ia [i]; k < ia [i+1]; k++) {

		    T value = a[k];
	    	    if (logscale) value = filter(value);
		    int ic = tab.color_index(value);
		    s << "c" << ic << " " 
		      << ja [k]    << " " 
		      << nrow-i-1  << " p\n";
	    }
	} else { 

	    // black & white: plotting loop
            for (Index i = 0; i < nrow; i++)
                for (Index k = ia [i]; k < ia [i+1]; k++)

                    s << ja [k] << " " << nrow-i-1 << " p\n";
	}
	s << "showpage\n";
}
}// namespace rheolef
# endif // _SKIT_OUTPS_H
