///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// mass matrix on a surface, as defined by a level set
//
#include "mass_s.h"
#include "rheolef/ublas_matrix_range.h"

namespace rheolef {
extern void compute_matrix_m (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk);
extern void compute_matrix_extension (const geo_element& K, const ublas::vector<point>& x, const ublas::vector<Float>& f, ublas::matrix<Float>& mk);
extern bool belongs_to_band (const geo_element& K, const ublas::vector<Float>& f);
} // namespace rheolef

using namespace rheolef;
using namespace std;
using namespace ublas;

void 
mass_s::operator() (const geo_element& K, ublas::matrix<Float>& m) const
{
    size_type nloc = K.size();
    ublas::vector<point> x (nloc);
    ublas::vector<Float> f (nloc);
    tiny_vector<size_t> bgd_idof (nloc);
    const field& phi_h = _wh;
    space Bh;
    switch (get_XY_type()) {
      case Bh_Bh: // Bh is used only to get the mesh from, so no need to check whether first space is P0 or P1
      case Bh_Wh: Bh = get_first_space(); break;
      case Wh_Bh:
      default:    Bh = get_second_space(); break;
    }
    const geo& beta_h   = Bh.get_geo();
    const geo& lambda_h = Bh.get_global_geo();
    phi_h.get_space().set_global_dof (K, bgd_idof);
    for (size_t iloc = 0; iloc < nloc; iloc++) {
      x[iloc] = lambda_h.vertex(K[iloc]);
      f[iloc] = phi_h.at (bgd_idof[iloc]);
    }
    if (!belongs_to_band(K, f)) {
        // empty m matrix => no assembly peformed.
	m.resize(0,0);
        return;
    }
    if (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() &&
        get_second_space().dimension() == get_second_space().get_geo().map_dimension()) {

      // form ("mass_s", Bh, Bh, phi_h):
      m.resize (nloc, nloc);
      compute_matrix_m (K, x, f, m);

      // then aggregate matrix in case it's a P0 matrix which is wanted:
      if (get_second_space().get_approx()=="P0")
       {
	 ublas::matrix<Float> ma (1, nloc, 0);
	 for (size_t j=0; j<nloc; j++) 
	  { 
//	    ma(0, j)=0;
	    for (size_t i=0; i<nloc; i++) ma(0, j) += m(i,j);
	  }
	 m.resize(1, nloc);
	 for (size_t j=0; j<nloc; j++) m(0,j)=ma(0,j);
       }

      if (get_first_space().get_approx()=="P0")
       {
         size_t ni = m.size1();
	 ublas::matrix<Float> ma (ni, 1, 0);
	 for (size_t i=0; i<ni; i++) 
	  { 
//	    ma(0, j)=0;
	    for (size_t j=0; j<nloc; j++) ma(i, 0) += m(i,j);
	  }
	 m.resize(ni, 1);
	 for (size_t i=0; i<ni; i++) m(i,0)=ma(i,0);
       }

    } else if (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() +1 &&
               get_second_space().dimension() == get_second_space().get_geo().map_dimension()) {

      // form ("mass_s", Wh, Bh, phi_h): extension
      m.resize (nloc, nloc-1);
      compute_matrix_extension (K, x, f, m);

    } else if (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() &&
               get_second_space().dimension() == get_second_space().get_geo().map_dimension()+1) {

      // form ("mass_s", Bh, Wh, phi_h): projection
      ublas::matrix<Float> mt (nloc, nloc-1);
      compute_matrix_extension (K, x, f, mt);
      m.resize (nloc-1, nloc);
      for (size_t i = 0; i < nloc-1; i++) {
      for (size_t j = 0; j < nloc;   j++) {
	m(i,j) = mt(j,i);
      }}
    } else {
      error_macro ("unexpected dimensions and map_dimensions: not yet implemented.");
    }
}
void
mass_s::check_after_initialize () const
{
  // suppose also that multi-component spaces are homogeneous,
  // i.e. that all components have the same approx
  check_macro (get_first_space().n_component() == 1,
    "incompatible spaces for the `mass_s' form.");
  check_macro (get_first_space().n_component() == get_second_space().n_component(),
    "incompatible spaces for the `mass_s' form.");
  check_macro (get_first_space().get_approx() == "P1" || get_first_space().get_approx() == "P0", 
  	"`mass_s' form: P0 or P1 approx required");
  check_macro (get_second_space().get_approx() == "P1" || get_second_space().get_approx() == "P0", 
  	"`mass_s' form: P0 or P1 approx required");
  check_macro (is_weighted(), "`mass_s' form may specify the level set function");

  if        (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() +1 &&
             get_second_space().dimension() == get_second_space().get_geo().map_dimension()) {
       _XY_switch = Wh_Bh; // extension
  } else if (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() &&
             get_second_space().dimension() == get_second_space().get_geo().map_dimension()+1) {
       _XY_switch = Bh_Wh; // prolongement
  } else if (get_first_space().dimension()  == get_first_space().get_geo().map_dimension() &&
             get_second_space().dimension() == get_second_space().get_geo().map_dimension()) {
       _XY_switch = Bh_Bh; // classic banded level set
  } else {
       error_macro ("form on a banded level set: at least one of the two spaces may be a band");
  }
}
