#!vtk
#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
#
# mud flow visualization
#
# value render for scalar data on unstructured 2d meshes
# in polydata format
#
# date: 2 may 2001
#
# author: Pierre.Saramito@imag.fr
#
# NOTE: tested on vtk-3.2
# 
# ----------------------------------------------------------
# usage
# ----------------------------------------------------------
#
# for inlusion in a 'wrapper.tcl' file:
#
#	set field_position  0
#	set n_field_per_step 1
#       set file_name "my-file.vtk"
#	source vtk_interactor.tcl
#	source vtk_volume_interactor.tcl
#	source vtk_animate_elevation_height.tcl
# then
#       vtk wrapper.tcl
#
# -------------------------------------------------------------------------
# implementation notes : z(x,y) and h(x,y) simultaneously
# -------------------------------------------------------------------------
# vtkThresholdTextureCoords is a filter that generates
# texture coordinates for any input dataset type given a threshold
# criterion. The criterion can take three forms: 
#	1) greater than a particular value (ThresholdByUpper()); #
#	2 less than a particular value (ThresholdByLower(); 
#	3) between two values (ThresholdBetween().
#
# If the threshold criterion is satisfied, the "in" texture coordinate will be set
# (this can be specified by the user).
# If the threshold criterion is not satisfied the "out" is set. 
#
# Warning: 
#  There is a texture map - texThres.vtk - that can be used in conjunction with this filter.
#  This map defines a "transparent" region for texture coordinates 0<=r<0.5, 
#  and an opaque full intensity map for texture coordinates 0.5<r<=1.0.
#  There is a small transition region for r=0.5. 
#
# -------------------------------------------------------------------------
# animation :
# -------------------------------------------------------------------------
#   many scalars : kitchen.vtk office.vtk blow.vtk
#	-> office.tcl officeStreamPoints.tcl officeTube.tcl
#	   streamPolygon.tcl
#	-> blow.tcl extractUGrid.tcl fieldToUGrid.tcl meshEdges.tcl
#	   meshEdges2.tcl pointToCellData.tcl 
#	   imaging/parallelCoordinates.tcl
#
#   scan *.vtk: polyEx2.tcl
# -----------------------------------------------------
# parameters
# -----------------------------------------------------

set mud_opacity 1.0
set epsilon     0.01;	# threshoold = hmin+epsilon
set axis_color "0. 0. 0."

# -----------------------------------------------------
# render window
# -----------------------------------------------------

vtkRenderer render
    render SetBackground 1 1 1

# -----------------------------------------------------
# reader and bounds
# -----------------------------------------------------

vtkPolyDataReader reader
    reader SetFileName $file_name
    reader Update

if {[get_vtk_major_version] >= 3 && [get_vtk_minor_version] >= 2} {
            # scan data for animations
            set n_field [reader GetNumberOfScalarsInFile]
            for {set i 0} {$i < $n_field} {incr i} {
                set name($i) [reader GetScalarsNameInFile $i]
                
            }
} else {
            # scan with a "grep SCALARS"
            set grep_output [eval exec [list grep SCALARS $file_name]]
            set i 0
            foreach {scalars name_i float} $grep_output {
                set name($i) $name_i
                incr i
            }
            set n_field $i
}
reader SetScalarsName $name(0)

set n_step [expr $n_field / $n_field_per_step]
set  bounds [[reader GetOutput] GetBounds]
scan $bounds "%f %f %f %f %f %f" minx maxx miny maxy minz maxz

set xmid [expr ($maxx + $minx)/2.0];
set ymid [expr ($maxy + $miny)/2.0];
set zmid [expr ($maxz + $minz)/2.0];
set xdiff [expr $maxx - $minx];
set ydiff [expr $maxy - $miny];
set zdiff [expr $maxz - $minz];

set hrange [[[[reader GetOutput] GetPointData] GetScalars] GetRange]
set hmin [lindex $hrange 0]
set hmax [lindex $hrange 1]

# -----------------------------------------------------
# mud height color table
# -----------------------------------------------------

vtkLookupTable color_table
    color_table SetHueRange        0.6 1.0
    color_table SetSaturationRange 0.5 1.0
    color_table SetValueRange      1.0 1.0

# -----------------------------------------------------
# mud height rendering : h(x,y)
# -----------------------------------------------------

vtkThresholdTextureCoords threshold
    threshold SetInput [reader GetOutput]
    eval threshold ThresholdByUpper [expr $hmin + $epsilon]

# add : z(x,y) + h(x,y)
vtkWarpScalar apply_height
  apply_height SetInput [threshold GetOutput]
  apply_height SetNormal 0 0 1
  apply_height SetScaleFactor 1
  apply_height UseNormalOn

vtkPolyDataNormals value_normals
    value_normals SetInput [apply_height GetOutput]
    value_normals SetFeatureAngle 60

vtkDataSetMapper value_mapper
    value_mapper SetInput [value_normals GetOutput]
    #eval value_mapper SetScalarRange [[threshold GetOutput] GetScalarRange]
    eval value_mapper SetScalarRange $hrange
    value_mapper SetLookupTable color_table
    value_mapper ImmediateModeRenderingOn

vtkStructuredPointsReader texture_map
  eval texture_map SetFileName $texture_file

vtkTexture texture
  texture SetInput [texture_map GetOutput]
  texture InterpolateOff
  texture RepeatOff

vtkActor value_actor
    value_actor SetMapper value_mapper
    value_actor SetTexture texture
    eval [value_actor GetProperty] SetOpacity $mud_opacity

render AddActor value_actor

# -----------------------------------------------------
# geographic color table
# -----------------------------------------------------

vtkLookupTable geographic_color_table
    geographic_color_table SetHueRange        0.4 0.0
    geographic_color_table SetSaturationRange 1.0 0.1
    geographic_color_table SetValueRange      0.5 1.0

# -----------------------------------------------------
# elevation rendering : z(x,y) where h(x,y) <= h0
# -----------------------------------------------------

# do not draw top where there is mud with opacity=1
#vtkThresholdTextureCoords low_threshold
#    low_threshold SetInput [reader GetOutput]
#    eval low_threshold ThresholdByLower [expr $hmin + $epsilon]
#
vtkElevationFilter elevation
    elevation SetInput [reader GetOutput]
    eval elevation SetLowPoint 0 0 $minz
    eval elevation SetHighPoint 0 0 $maxz
    eval elevation SetScalarRange $minz $maxz

vtkPolyDataNormals elevation_normals
    elevation_normals SetInput [elevation GetOutput]
    elevation_normals SetFeatureAngle 60

vtkDataSetMapper elevation_mapper
    elevation_mapper SetInput [elevation_normals GetOutput]
    eval elevation_mapper SetScalarRange $minz $maxz
    elevation_mapper SetLookupTable geographic_color_table
    elevation_mapper ImmediateModeRenderingOn

vtkActor elevation_actor
    elevation_actor SetMapper elevation_mapper
# no texture if no filter
#   elevation_actor SetTexture texture

render AddActor elevation_actor

# -----------------------------------------------------
# outline edges
# -----------------------------------------------------

vtkCubeSource outline
    outline SetXLength $xdiff;
    outline SetYLength $ydiff;
    outline SetZLength $zdiff;
    outline SetCenter  [ expr $xmid ] \
                       [ expr $ymid ] \
                       [ expr $zmid ]

vtkExtractEdges outline_edge
    outline_edge SetInput [outline GetOutput]

vtkPolyDataMapper outline_mapper
    outline_mapper SetInput [outline_edge GetOutput]
    outline_mapper ImmediateModeRenderingOn

vtkActor outline_actor
    outline_actor SetMapper outline_mapper
    eval [outline_actor GetProperty] SetColor    $axis_color
    [outline_actor GetProperty] SetAmbient  1
    [outline_actor GetProperty] SetDiffuse  0
    [outline_actor GetProperty] SetSpecular 0
    [outline_actor GetProperty] SetRepresentationToWireframe

render AddActor  outline_actor

# -----------------------------------------------------
# axis
# -----------------------------------------------------

vtkCubeAxesActor2D axis_actor
      axis_actor SetInput [reader GetOutput]
      axis_actor SetCamera [render GetActiveCamera]
      axis_actor SetLabelFormat "%6.4g"
      if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
	  vtkTextProperty axis_actor_property
	  axis_actor_property ShadowOn
	  axis_actor SetAxisLabelTextProperty axis_actor_property
	  axis_actor SetAxisTitleTextProperty axis_actor_property
      } else {
	  axis_actor ShadowOn
      }
      axis_actor SetFlyModeToOuterEdges
      axis_actor SetFontFactor 0.8
      eval [axis_actor GetProperty] SetColor $axis_color

render AddProp axis_actor

# -----------------------------------------------------
# color bar
# -----------------------------------------------------

vtkScalarBarActor bar_actor
    #bar_actor SetLookupTable [value_mapper GetLookupTable]
    bar_actor SetLookupTable color_table
    bar_actor SetOrientationToHorizontal
    [bar_actor GetProperty] SetColor 0 0 0
    [bar_actor GetPositionCoordinate] SetCoordinateSystemToNormalizedViewport
    [bar_actor GetPositionCoordinate] SetValue 0.1 0.01
    bar_actor SetOrientationToHorizontal
    bar_actor SetWidth  0.8
    bar_actor SetHeight 0.10
    bar_actor SetTitle "mud height"
    if {[get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
        vtkTextProperty bar_text_property
        bar_text_property BoldOn
	bar_actor SetTitleTextProperty bar_text_property
	bar_actor SetLabelTextProperty bar_text_property
    } else {
        bar_actor BoldOn
    }
    eval [bar_actor GetProperty] SetColor $axis_color

render AddActor bar_actor

# -----------------------------------------------------
# rendering all
# -----------------------------------------------------

vtkRenderWindow render_window
    render_window AddRenderer render
    render_window SetSize 500 500

vtkRenderWindowInteractor interactor
    interactor SetRenderWindow render_window
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
	iren AddObserver UserEvent {wm deiconify .vtkInteract};
    } else {
	iren SetUserMethod {wm deiconify .vtkInteract};
    }
    interactor Initialize

# position of the camera
[render GetActiveCamera] Zoom 1.4
[render GetActiveCamera] Roll -45	;# Rotate in xy plane
[render GetActiveCamera] Elevation 91	;# Rotate also

render_window Render

# -----------------------------------------------------
# loop on all scalar fields
# -----------------------------------------------------
proc start {} {
  global field_position
  global n_field_per_step
  global n_field
  global name
  set total_cpu 0
  for {set i $field_position} {$i < $n_field} {incr i $n_field_per_step} {
    set i_name "$name($i)"
    reader SetScalarsName $i_name
    bar_actor SetTitle "mud height : step $i : ${i_name}"
    render_window Render
    set cpu [render GetLastRenderTimeInSeconds]
    set total_cpu [expr $total_cpu + $cpu]
  }
}
# -----------------------------------------------------
# graphic user interaction
# -----------------------------------------------------
toplevel .control
#pack .control

button .control.start -text start -fg #999999 -bg #440066 \
        -activeforeground #440066 -activebackground #999999 \
        -highlightthickness 0 -bd 4 -command { start } \
        -font {Helvetica -12 bold}

pack .control.start 

# -----------------------------------------------------
# action
# -----------------------------------------------------
wm title . $file_name
wm withdraw .

