# SPDX-License-Identifier: Apache-2.0
# Copyright 2017, 2019 The Meson development team

from __future__ import annotations

import os
import typing as T

from . import ExtensionModule, ModuleInfo
from .. import mesonlib
from ..interpreterbase import noKwargs, typed_pos_args

if T.TYPE_CHECKING:
    from ..interpreter import Interpreter
    from . import ModuleState

class KeyvalModule(ExtensionModule):

    INFO = ModuleInfo('keyval', '0.55.0', stabilized='0.56.0')

    def __init__(self, interp: 'Interpreter'):
        super().__init__(interp)
        self.methods.update({
            'load': self.load,
        })

    @staticmethod
    def _load_file(path_to_config: str) -> T.Dict[str, str]:
        result: T.Dict[str, str] = {}
        try:
            with open(path_to_config, encoding='utf-8') as f:
                for line in f:
                    if '#' in line:
                        comment_idx = line.index('#')
                        line = line[:comment_idx]
                    line = line.strip()
                    try:
                        name, val = line.split('=', 1)
                    except ValueError:
                        continue
                    result[name.strip()] = val.strip()
        except OSError as e:
            raise mesonlib.MesonException(f'Failed to load {path_to_config}: {e}')

        return result

    @noKwargs
    @typed_pos_args('keyval.load', (str, mesonlib.File))
    def load(self, state: 'ModuleState', args: T.Tuple['mesonlib.FileOrString'], kwargs: T.Dict[str, T.Any]) -> T.Dict[str, str]:
        s = args[0]
        is_built = False
        if isinstance(s, mesonlib.File):
            is_built = is_built or s.is_built
            s = s.absolute_path(self.interpreter.environment.source_dir, self.interpreter.environment.build_dir)
        else:
            s = os.path.join(self.interpreter.environment.source_dir, s)

        if not is_built:
            self.interpreter.build_def_files.add(s)

        return self._load_file(s)


def initialize(interp: 'Interpreter') -> KeyvalModule:
    return KeyvalModule(interp)
