/**
 * @file EventTypes.hh
 *
 * @brief   Template event classes (inherit from EventBase)
 *
 * @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
 *
 * @section Licence
 *   |LICENCE|
 */


#ifndef _EVENTTYPES_HH_
#define _EVENTTYPES_HH_

#include <iostream>
#include <sstream>
#include <string>
#include "EventBase.hh"
#include "EventManager.hh"

namespace events {
/**
 * @class EventFrom
 * Used for all events generated by one object of the model about its own state
 * @param SOURCE_TYPE   the class of the model object
 *                      --> must implement string toString() const
 */

template <class SOURCE_TYPE, class MSG_CLASS>
class EventFrom : public EventBase {
public:
  EventFrom(EventBase::Severity severity, std::string message,
            std::string data, const SOURCE_TYPE * src)
  : EventBase(severity), myMessage(message), myData(data), mySource(src) {
  }

  std::string
  getMessage() const {
    return myMessage;
  }

  std::string
  getData() const {
    return myData;
  }

  const SOURCE_TYPE *
  getSource() const {
    return mySource;
  }

  virtual std::string
  toString() const;

private:
  std::string myMessage;
  std::string myData;
  const SOURCE_TYPE *mySource;
  static const int myMsgCode;
};

template <class SOURCE_TYPE, class MSG_CLASS>
std::string
EventFrom<SOURCE_TYPE, MSG_CLASS>::toString() const {
  std::ostringstream str;
  str << EventBase::toString() << " : " << myMessage << " : ";
  str << "FROM " << mySource->toString() << " : " << myData << "/END";
  return str.str();
}

/**
 * @class EventStandardMsg
 * @param SOURCE_TYPE   the type of object producing the event
 * @param MSG_TYPE      integer code for the message type (declared in source obj class)
 */
template <class SOURCE_TYPE, int MSG_TYPE>
class EventStandardMsg {
private:
  static const int myMsgCode;
};

template <class SOURCE_TYPE, int MSG_TYPE>
const int
EventStandardMsg<SOURCE_TYPE, MSG_TYPE>::myMsgCode = MSG_TYPE;


template<class SOURCE_TYPE, int MSG_TYPE>
static void
sendEventFrom(const SOURCE_TYPE *src, std::string msg,
              std::string data, EventBase::Severity severity) {
  EventManager::getEventMgr()->sendEvent(
    new EventFrom<SOURCE_TYPE,
                  EventStandardMsg<SOURCE_TYPE, MSG_TYPE> >(severity, msg,
                                                            data, src));
}

/**
 * @class EventCreateObject
 * @param OBJ_TYPE      the type of object created
 * @param PARENT_TYPE   the type of the parent of the object created
 */

template <class OBJ_TYPE, class PARENT_TYPE>
class EventCreateObject : public EventBase {
public:
  EventCreateObject(OBJ_TYPE * obj, PARENT_TYPE * parent)
  : myObj(obj), myParent(parent) {
  }

  const OBJ_TYPE *
  getObject() const {
    return myObj;
  }

  const PARENT_TYPE *
  getParent() const {
    return myParent;
  }

  virtual std::string
  toString() const;

private:
  OBJ_TYPE *myObj;
  PARENT_TYPE *myParent;
};

template <class OBJ_TYPE, class PARENT_TYPE>
std::string
EventCreateObject<OBJ_TYPE, PARENT_TYPE>::toString() const {
  if (myParent == NULL) {
    return "CREATION OF " + myObj->toString();
  } else {
    return "CREATION OF " + myObj->toString() +
           " [PARENT=" + myParent->toString() + "]";
  }
}


/**
 * @class EventCreateDirectedLink
 * @param SRC_TYPE      the type of the source object
 * @param DEST_TYPE     the type of the destination object
 */
template <class SRC_TYPE, class DEST_TYPE>
class EventCreateDirectedLink : public EventBase {
public:
  EventCreateDirectedLink(SRC_TYPE * src, DEST_TYPE * dest)
  : mySrc(src), myDest(dest) {
  }

  const SRC_TYPE *
  getSource() const {
    return mySrc;
  }

  const DEST_TYPE *
  getDestination() const {
    return myDest;
  }

  virtual std::string
  toString() const;

private:
  SRC_TYPE *mySrc;
  DEST_TYPE *myDest;
};

template <class SRC_TYPE, class DEST_TYPE>
std::string
EventCreateDirectedLink<SRC_TYPE, DEST_TYPE>::toString() const {
  return "CREATION OF LINK FROM=" + mySrc->toString() +
         " TO=" + myDest->toString();
}
}
#endif  // _WFEVENT_HH_
