/*
     This file is part of GNUnet
     (C) 2011, 2012 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/fs/gnunet-fs-gtk_main-window-search.c
 * @author Christian Grothoff
 * @brief event handlers for the search function in the main window
 */
#include "gnunet-fs-gtk_common.h"
#include "gnunet-fs-gtk.h"
#include "gnunet-fs-gtk_anonymity-widgets.h"


/**
 * Start a search.
 *
 * @param builder the main dialog builder
 */
static void
start_search (struct GNUNET_GTK_MainWindowContext *main_ctx)
{
  guint anonymity_level;
  gchar *keywords;
  gchar *mime_keyword;
  GNUNET_HashCode *nsid;
  struct GNUNET_FS_Uri *uri;

  /* get anonymity level */
  if (!GNUNET_GTK_get_selected_anonymity_level
      (main_ctx->builder, "main_window_search_anonymity_combobox", &anonymity_level))
  {
    GNUNET_break (0);
    return;
  }

  /* get selected mime type */
  {
    GtkComboBox *mime_combo;
    GtkTreeModel *mime_model;
    GtkTreeIter iter;

    mime_combo = GTK_COMBO_BOX (GNUNET_FS_GTK_get_main_window_object
				("main_window_search_mime_combobox"));
    mime_model = gtk_combo_box_get_model (mime_combo);
    mime_keyword = NULL;
    if ( (NULL != mime_model) && 
	 gtk_combo_box_get_active_iter (mime_combo, &iter))
      gtk_tree_model_get (mime_model, &iter,
                          GNUNET_GTK_FS_MAIN_WINDOW_SEARCH_MIME_MC_MIME,
                          &mime_keyword,
                          -1);
    if ( (NULL != mime_keyword) &&
	 (0 == strcmp (mime_keyword, " ")) )
    {
      g_free (mime_keyword);
      mime_keyword = NULL;
    }
  }

  /* get selected namespace */
  {
    GtkTreeRowReference *ref;
    GtkTreeIter iter;

    nsid = NULL;    
    ref = g_object_get_data (G_OBJECT (main_ctx->ns_dropdown_button), "selected-row-reference");
    if (NULL != ref)
    {
      GtkTreePath *namespace_treepath;
      GtkTreeModel *namespace_model;
      
      namespace_model = gtk_tree_row_reference_get_model (ref);
      namespace_treepath = gtk_tree_row_reference_get_path (ref);
      if ( (NULL != namespace_treepath) &&
	   (gtk_tree_model_get_iter (namespace_model, &iter, namespace_treepath)))
        gtk_tree_model_get (namespace_model, &iter,
                            GNUNET_GTK_FS_MAIN_WINDOW_SEARCH_MIME_MC_TYPE,
                            &nsid,
                            -1);
    }
  }
   
  /* get keywords and compose keyword string */
  {
    const char *entry_keywords;

    entry_keywords = gtk_entry_get_text (main_ctx->search_entry);  
    if (NULL != mime_keyword)
    {
      keywords = g_strdup_printf ("%s +%s", 
				  entry_keywords, 
				  mime_keyword);
      g_free (mime_keyword);
    }
    else
    {
      keywords = g_strdup (entry_keywords);
    }
  }

  /* build KSK/SKS URI */
  if (NULL != nsid)
  {
    uri = GNUNET_FS_uri_sks_create_from_nsid (nsid, keywords);
    GNUNET_assert (uri != NULL);
  }
  else
  {
    char *emsg = NULL;

    uri = GNUNET_FS_uri_ksk_create (keywords, &emsg);
    if (NULL == uri)
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR, _("Invalid keyword string `%s': %s"),
                  keywords, emsg);
      g_free (keywords);
      GNUNET_free_non_null (emsg);
      return;
    }
  }
  g_free (keywords);

  /* start search */
  GNUNET_FS_search_start (GNUNET_FS_GTK_get_fs_handle (), 
			  uri, anonymity_level,
                          GNUNET_FS_SEARCH_OPTION_NONE, NULL);
  GNUNET_FS_uri_destroy (uri);
}


/**
 * User clicked on the 'search' button in the main window.
 *
 * @param button the search button
 * @param user_data the main dialog builder
 */
void
main_window_search_button_clicked_cb (GtkButton * button, 
				      gpointer user_data)
{
  struct GNUNET_GTK_MainWindowContext *main_ctx = user_data;
  start_search (main_ctx);
}


/**
 * User pushed a key (possibly ENTER) in the search entry widget.
 * Start the search if it was ENTER.
 *
 * @param widget the entry widget
 * @param event the key stroke
 * @param user_data the builder of the main window
 * @return FALSE if this was not ENTER, TRUE if it was
 */
gboolean
main_window_search_entry_key_press_event_cb (GtkWidget * widget,
                                             GdkEventKey * event,
                                             gpointer user_data)
{
  struct GNUNET_GTK_MainWindowContext *main_ctx = user_data;

  if (event->keyval == GDK_KEY_Return)
  {
    start_search (main_ctx);
    return TRUE;
  }
  return FALSE;
}

/* end of gnunet-fs-gtk_main-window-search.c */
