/* ----------------------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   Steve Plimpton, sjplimp@sandia.gov

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#include "../testing/core.h"
#include "atom.h"
#include "atom_vec_body.h"
#include "atom_vec_ellipsoid.h"
#include "atom_vec_hybrid.h"
#include "atom_vec_line.h"
#include "atom_vec_tri.h"
#include "body.h"
#include "input.h"
#include "lammps.h"
#include "math_const.h"
#include "utils.h"
#include "gmock/gmock.h"
#include "gtest/gtest.h"

#include <cmath>
#include <cstdio>
#include <cstring>
#include <mpi.h>
#include <vector>

#if !defined(_FORTIFY_SOURCE) || (_FORTIFY_SOURCE == 0)
#if defined(__INTEL_COMPILER) || (__PGI)
#define _do_nothing
#elif defined(__clang__)
#pragma clang optimize off
#elif defined(__GNUC__)
#if (__GNUC__ > 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 9))
#pragma GCC optimize("no-var-tracking-assignments", "O0")
#endif
#endif
#endif

#define GETIDX(i) lmp->atom->map(i)

using LAMMPS_NS::utils::split_words;

static void create_molecule_files(const std::string &h2o_filename, const std::string &co2_filename)
{
    // create molecule files
    const char h2o_file[] = "# Water molecule. SPC/E model.\n\n3 atoms\n2 bonds\n1 angles\n\n"
                            "Coords\n\n1 1.12456 0.09298 1.27452\n"
                            "2 1.53683 0.75606 1.89928\n3 0.49482 0.56390 0.65678\n\n"
                            "Types\n\n1 1\n2 2\n3 2\n\n"
                            "Charges\n\n1 -0.8472\n2 0.4236\n3 0.4236\n\n"
                            "Bonds\n\n1 1 1 2\n2 1 1 3\n\n"
                            "Angles\n\n1 1 2 1 3\n\n"
                            "Shake Flags\n\n1 1\n2 1\n3 1\n\n"
                            "Shake Atoms\n\n1 1 2 3\n2 1 2 3\n3 1 2 3\n\n"
                            "Shake Bond Types\n\n1 1 1 1\n2 1 1 1\n3 1 1 1\n\n"
                            "Special Bond Counts\n\n1 2 0 0\n2 1 1 0\n3 1 1 0\n\n"
                            "Special Bonds\n\n1 2 3\n2 1 3\n3 1 2\n\n";
    const char co2_file[] = "# CO2 molecule file. TraPPE model.\n\n"
                            "3 atoms\n2 bonds\n1 angles\n\n"
                            "Coords\n\n1 0.0 0.0 0.0\n2 -1.16 0.0 0.0\n3 1.16 0.0 0.0\n\n"
                            "Types\n\n1 1\n2 2\n3 2\n\n"
                            "Charges\n\n1 0.7\n2 -0.35\n3 -0.35\n\n"
                            "Bonds\n\n1 1 1 2\n2 1 1 3\n\n"
                            "Angles\n\n1 1 2 1 3\n\n"
                            "Special Bond Counts\n\n1 2 0 0\n2 1 1 0\n3 1 1 0\n\n"
                            "Special Bonds\n\n1 2 3\n2 1 3\n3 1 2\n\n";

    FILE *fp = fopen(h2o_filename.c_str(), "w");
    if (fp) {
        fputs(h2o_file, fp);
        fclose(fp);
    }
    fp = fopen(co2_filename.c_str(), "w");
    if (fp) {
        fputs(co2_file, fp);
        fclose(fp);
    }
}

// whether to print verbose output (i.e. not capturing LAMMPS screen output).
bool verbose = false;

const double EPSILON = 5.0e-14;

namespace LAMMPS_NS {
using ::testing::Eq;

class AtomStyleTest : public LAMMPSTest {
protected:
    static void SetUpTestSuite() { create_molecule_files("h2o.mol", "co2.mol"); }

    static void TearDownTestSuite()
    {
        remove("h2o.mol");
        remove("co2.mol");
    }

    void SetUp() override
    {
        testbinary = "AtomStyleTest";
        LAMMPSTest::SetUp();
        ASSERT_NE(lmp, nullptr);
        BEGIN_HIDE_OUTPUT();
        command("units real");
        command("dimension 3");
        command("pair_style zero 4.0");
        command("region box block -4 4 -4 4 -4 4");
        END_HIDE_OUTPUT();
    }

    void TearDown() override
    {
        LAMMPSTest::TearDown();
        remove("test_atom_styles.data");
        remove("input_atom_styles.data");
        remove("test_atom_styles.restart");
    }
};

// default class Atom state
struct AtomState {
    std::string atom_style      = "atomic";
    bigint natoms               = 0;
    int nlocal                  = 0;
    int nghost                  = 0;
    int nmax                    = 1;
    int tag_enable              = 1;
    int molecular               = Atom::ATOMIC;
    bigint nellipsoids          = 0;
    bigint nlines               = 0;
    bigint ntris                = 0;
    bigint nbodies              = 0;
    bigint nbonds               = 0;
    bigint nangles              = 0;
    bigint ndihedrals           = 0;
    bigint nimpropers           = 0;
    int ntypes                  = 0;
    int nbondtypes              = 0;
    int nangletypes             = 0;
    int ndihedraltypes          = 0;
    int nimpropertypes          = 0;
    int bond_per_atom           = 0;
    int angle_per_atom          = 0;
    int dihedral_per_atom       = 0;
    int improper_per_atom       = 0;
    int extra_bond_per_atom     = 0;
    int extra_angle_per_atom    = 0;
    int extra_dihedral_per_atom = 0;
    int extra_improper_per_atom = 0;

    int sphere_flag                  = 0;
    int ellipsoid_flag               = 0;
    int line_flag                    = 0;
    int tri_flag                     = 0;
    int body_flag                    = 0;
    int peri_flag                    = 0;
    int electron_flag                = 0;
    int wavepacket_flag              = 0;
    int sph_flag                     = 0;
    int molecule_flag                = 0;
    int molindex_flag                = 0;
    int molatom_flag                 = 0;
    int q_flag                       = 0;
    int mu_flag                      = 0;
    int rmass_flag                   = 0;
    int radius_flag                  = 0;
    int omega_flag                   = 0;
    int torque_flag                  = 0;
    int angmom_flag                  = 0;
    int vfrac_flag                   = 0;
    int spin_flag                    = 0;
    int eradius_flag                 = 0;
    int ervel_flag                   = 0;
    int erforce_flag                 = 0;
    int cs_flag                      = 0;
    int csforce_flag                 = 0;
    int vforce_flag                  = 0;
    int ervelforce_flag              = 0;
    int etag_flag                    = 0;
    int rho_flag                     = 0;
    int esph_flag                    = 0;
    int cv_flag                      = 0;
    int vest_flag                    = 0;
    int dpd_flag                     = 0;
    int edpd_flag                    = 0;
    int tdpd_flag                    = 0;
    int mesont_flag                  = 0;
    int sp_flag                      = 0;
    int x0_flag                      = 0;
    int smd_flag                     = 0;
    int damage_flag                  = 0;
    int contact_radius_flag          = 0;
    int smd_data_9_flag              = 0;
    int smd_stress_flag              = 0;
    int eff_plastic_strain_flag      = 0;
    int eff_plastic_strain_rate_flag = 0;
    double pdscale                   = 1.0;

    int maxspecial = 1;

    int nmolecule = 0;

    int nivector = 0;
    int ndvector = 0;

    int nextra_grow        = 0;
    int nextra_restart     = 0;
    int nextra_border      = 0;
    int nextra_grow_max    = 0;
    int nextra_restart_max = 0;
    int nextra_border_max  = 0;
    int nextra_store       = 0;

    int map_style      = Atom::MAP_NONE;
    int map_user       = 0;
    tagint map_tag_max = -1;

    // properties X that aren't controlled by an equivalent X_flag
    bool has_type  = true;
    bool has_mask  = true;
    bool has_image = true;
    bool has_x     = true;
    bool has_v     = true;
    bool has_f     = true;

    bool has_bonds    = false;
    bool has_angles   = false;
    bool has_dihedral = false;
    bool has_improper = false;

    bool has_ivname       = false;
    bool has_dvname       = false;
    bool has_ianame       = false;
    bool has_daname       = false;
    bool has_mass         = false;
    bool has_mass_setflag = false;

    bool has_nspecial = false;
    bool has_special  = false;

    bool has_extra_grow    = false;
    bool has_extra_restart = false;
    bool has_extra_border  = false;
    bool has_extra         = false;
    bool has_sametag       = false;
};

#define ASSERT_ARRAY_ALLOCATED(ptr, enabled) \
    if (enabled) {                           \
        ASSERT_NE(ptr, nullptr);             \
    } else {                                 \
        ASSERT_EQ(ptr, nullptr);             \
    }

void ASSERT_ATOM_STATE_EQ(Atom *atom, const AtomState &expected)
{
    ASSERT_THAT(std::string(atom->atom_style), Eq(expected.atom_style));

    ASSERT_NE(atom->avec, nullptr);
    ASSERT_EQ(atom->natoms, expected.natoms);
    ASSERT_EQ(atom->nlocal, expected.nlocal);
    ASSERT_EQ(atom->nghost, expected.nghost);
    ASSERT_EQ(atom->nmax == 1, expected.nmax == 1);
    ASSERT_EQ(atom->tag_enable, expected.tag_enable);
    ASSERT_EQ(atom->molecular, expected.molecular);
    ASSERT_EQ(atom->nellipsoids, expected.nellipsoids);
    ASSERT_EQ(atom->nlines, expected.nlines);
    ASSERT_EQ(atom->ntris, expected.ntris);
    ASSERT_EQ(atom->nbodies, expected.nbodies);
    ASSERT_EQ(atom->nbonds, expected.nbonds);
    ASSERT_EQ(atom->nangles, expected.nangles);
    ASSERT_EQ(atom->ndihedrals, expected.ndihedrals);
    ASSERT_EQ(atom->nimpropers, expected.nimpropers);
    ASSERT_EQ(atom->ntypes, expected.ntypes);
    ASSERT_EQ(atom->nbondtypes, expected.nbondtypes);
    ASSERT_EQ(atom->nangletypes, expected.nangletypes);
    ASSERT_EQ(atom->ndihedraltypes, expected.ndihedraltypes);
    ASSERT_EQ(atom->nimpropertypes, expected.nimpropertypes);
    ASSERT_EQ(atom->bond_per_atom, expected.bond_per_atom);
    ASSERT_EQ(atom->angle_per_atom, expected.angle_per_atom);
    ASSERT_EQ(atom->dihedral_per_atom, expected.dihedral_per_atom);
    ASSERT_EQ(atom->improper_per_atom, expected.improper_per_atom);
    ASSERT_EQ(atom->extra_bond_per_atom, expected.extra_bond_per_atom);
    ASSERT_EQ(atom->extra_angle_per_atom, expected.extra_angle_per_atom);
    ASSERT_EQ(atom->extra_dihedral_per_atom, expected.extra_dihedral_per_atom);
    ASSERT_EQ(atom->extra_improper_per_atom, expected.extra_improper_per_atom);

    ASSERT_EQ(atom->sphere_flag, expected.sphere_flag);
    ASSERT_EQ(atom->ellipsoid_flag, expected.ellipsoid_flag);
    ASSERT_EQ(atom->line_flag, expected.line_flag);
    ASSERT_EQ(atom->tri_flag, expected.tri_flag);
    ASSERT_EQ(atom->body_flag, expected.body_flag);
    ASSERT_EQ(atom->peri_flag, expected.peri_flag);
    ASSERT_EQ(atom->electron_flag, expected.electron_flag);
    ASSERT_EQ(atom->wavepacket_flag, expected.wavepacket_flag);
    ASSERT_EQ(atom->sph_flag, expected.sph_flag);
    ASSERT_EQ(atom->molecule_flag, expected.molecule_flag);
    ASSERT_EQ(atom->molindex_flag, expected.molindex_flag);
    ASSERT_EQ(atom->molatom_flag, expected.molatom_flag);
    ASSERT_EQ(atom->q_flag, expected.q_flag);
    ASSERT_EQ(atom->mu_flag, expected.mu_flag);
    ASSERT_EQ(atom->rmass_flag, expected.rmass_flag);
    ASSERT_EQ(atom->radius_flag, expected.radius_flag);
    ASSERT_EQ(atom->omega_flag, expected.omega_flag);
    ASSERT_EQ(atom->torque_flag, expected.torque_flag);
    ASSERT_EQ(atom->angmom_flag, expected.angmom_flag);
    ASSERT_EQ(atom->vfrac_flag, expected.vfrac_flag);
    ASSERT_EQ(atom->spin_flag, expected.spin_flag);
    ASSERT_EQ(atom->eradius_flag, expected.eradius_flag);
    ASSERT_EQ(atom->ervel_flag, expected.ervel_flag);
    ASSERT_EQ(atom->erforce_flag, expected.erforce_flag);
    ASSERT_EQ(atom->cs_flag, expected.cs_flag);
    ASSERT_EQ(atom->csforce_flag, expected.csforce_flag);
    ASSERT_EQ(atom->vforce_flag, expected.vforce_flag);
    ASSERT_EQ(atom->ervelforce_flag, expected.ervelforce_flag);
    ASSERT_EQ(atom->etag_flag, expected.etag_flag);
    ASSERT_EQ(atom->rho_flag, expected.rho_flag);
    ASSERT_EQ(atom->esph_flag, expected.esph_flag);
    ASSERT_EQ(atom->cv_flag, expected.cv_flag);
    ASSERT_EQ(atom->vest_flag, expected.vest_flag);
    ASSERT_EQ(atom->dpd_flag, expected.dpd_flag);
    ASSERT_EQ(atom->edpd_flag, expected.edpd_flag);
    ASSERT_EQ(atom->tdpd_flag, expected.tdpd_flag);
    ASSERT_EQ(atom->mesont_flag, expected.mesont_flag);
    ASSERT_EQ(atom->sp_flag, expected.sp_flag);
    ASSERT_EQ(atom->x0_flag, expected.x0_flag);
    ASSERT_EQ(atom->smd_flag, expected.smd_flag);
    ASSERT_EQ(atom->damage_flag, expected.damage_flag);
    ASSERT_EQ(atom->contact_radius_flag, expected.contact_radius_flag);
    ASSERT_EQ(atom->smd_data_9_flag, expected.smd_data_9_flag);
    ASSERT_EQ(atom->smd_stress_flag, expected.smd_stress_flag);
    ASSERT_EQ(atom->eff_plastic_strain_flag, expected.eff_plastic_strain_flag);
    ASSERT_EQ(atom->eff_plastic_strain_rate_flag, expected.eff_plastic_strain_rate_flag);
    ASSERT_EQ(atom->pdscale, expected.pdscale);

    ASSERT_ARRAY_ALLOCATED(atom->tag, expected.tag_enable);
    ASSERT_ARRAY_ALLOCATED(atom->type, expected.has_type);
    ASSERT_ARRAY_ALLOCATED(atom->mask, expected.has_mask);
    ASSERT_ARRAY_ALLOCATED(atom->image, expected.has_image);

    ASSERT_ARRAY_ALLOCATED(atom->x, expected.has_x);
    ASSERT_ARRAY_ALLOCATED(atom->v, expected.has_v);
    ASSERT_ARRAY_ALLOCATED(atom->f, expected.has_f);
    ASSERT_ARRAY_ALLOCATED(atom->q, expected.q_flag);
    ASSERT_ARRAY_ALLOCATED(atom->mu, expected.mu_flag);

    ASSERT_ARRAY_ALLOCATED(atom->omega, expected.omega_flag);
    ASSERT_ARRAY_ALLOCATED(atom->angmom, expected.angmom_flag);
    ASSERT_ARRAY_ALLOCATED(atom->torque, expected.torque_flag);
    ASSERT_ARRAY_ALLOCATED(atom->radius, expected.radius_flag);
    ASSERT_ARRAY_ALLOCATED(atom->rmass, expected.rmass_flag);
    ASSERT_ARRAY_ALLOCATED(atom->ellipsoid, expected.ellipsoid_flag);
    ASSERT_ARRAY_ALLOCATED(atom->line, expected.line_flag);
    ASSERT_ARRAY_ALLOCATED(atom->tri, expected.tri_flag);
    ASSERT_ARRAY_ALLOCATED(atom->body, expected.body_flag);
    ASSERT_ARRAY_ALLOCATED(atom->molecule, expected.molecule_flag);
    ASSERT_ARRAY_ALLOCATED(atom->molindex, expected.molindex_flag);
    ASSERT_ARRAY_ALLOCATED(atom->molatom, expected.molatom_flag);

    ASSERT_ARRAY_ALLOCATED(atom->num_bond, expected.has_bonds);
    ASSERT_ARRAY_ALLOCATED(atom->bond_type, expected.has_bonds);
    ASSERT_ARRAY_ALLOCATED(atom->bond_atom, expected.has_bonds);

    ASSERT_ARRAY_ALLOCATED(atom->num_angle, expected.has_angles);
    ASSERT_ARRAY_ALLOCATED(atom->angle_type, expected.has_angles);
    ASSERT_ARRAY_ALLOCATED(atom->angle_atom1, expected.has_angles);
    ASSERT_ARRAY_ALLOCATED(atom->angle_atom2, expected.has_angles);
    ASSERT_ARRAY_ALLOCATED(atom->angle_atom3, expected.has_angles);

    ASSERT_ARRAY_ALLOCATED(atom->num_dihedral, expected.has_dihedral);
    ASSERT_ARRAY_ALLOCATED(atom->dihedral_type, expected.has_dihedral);
    ASSERT_ARRAY_ALLOCATED(atom->dihedral_atom1, expected.has_dihedral);
    ASSERT_ARRAY_ALLOCATED(atom->dihedral_atom2, expected.has_dihedral);
    ASSERT_ARRAY_ALLOCATED(atom->dihedral_atom3, expected.has_dihedral);
    ASSERT_ARRAY_ALLOCATED(atom->dihedral_atom4, expected.has_dihedral);

    ASSERT_ARRAY_ALLOCATED(atom->num_improper, expected.has_improper);
    ASSERT_ARRAY_ALLOCATED(atom->improper_type, expected.has_improper);
    ASSERT_ARRAY_ALLOCATED(atom->improper_atom1, expected.has_improper);
    ASSERT_ARRAY_ALLOCATED(atom->improper_atom2, expected.has_improper);
    ASSERT_ARRAY_ALLOCATED(atom->improper_atom3, expected.has_improper);
    ASSERT_ARRAY_ALLOCATED(atom->improper_atom4, expected.has_improper);

    ASSERT_EQ(atom->maxspecial, expected.maxspecial);

    ASSERT_ARRAY_ALLOCATED(atom->nspecial, expected.has_nspecial);
    ASSERT_ARRAY_ALLOCATED(atom->special, expected.has_special);

    // currently ignored
    ASSERT_ARRAY_ALLOCATED(atom->vfrac, false);
    ASSERT_ARRAY_ALLOCATED(atom->s0, false);
    ASSERT_ARRAY_ALLOCATED(atom->x0, false);
    ASSERT_ARRAY_ALLOCATED(atom->sp, false);
    ASSERT_ARRAY_ALLOCATED(atom->fm, false);
    ASSERT_ARRAY_ALLOCATED(atom->fm_long, false);
    ASSERT_ARRAY_ALLOCATED(atom->spin, false);
    ASSERT_ARRAY_ALLOCATED(atom->eradius, false);
    ASSERT_ARRAY_ALLOCATED(atom->ervel, false);
    ASSERT_ARRAY_ALLOCATED(atom->erforce, false);
    ASSERT_ARRAY_ALLOCATED(atom->ervelforce, false);
    ASSERT_ARRAY_ALLOCATED(atom->cs, false);
    ASSERT_ARRAY_ALLOCATED(atom->csforce, false);
    ASSERT_ARRAY_ALLOCATED(atom->vforce, false);
    ASSERT_ARRAY_ALLOCATED(atom->etag, false);
    ASSERT_ARRAY_ALLOCATED(atom->uCond, false);
    ASSERT_ARRAY_ALLOCATED(atom->uMech, false);
    ASSERT_ARRAY_ALLOCATED(atom->uChem, false);
    ASSERT_ARRAY_ALLOCATED(atom->uCG, false);
    ASSERT_ARRAY_ALLOCATED(atom->uCGnew, false);
    ASSERT_ARRAY_ALLOCATED(atom->duChem, false);
    ASSERT_ARRAY_ALLOCATED(atom->dpdTheta, false);
    ASSERT_ARRAY_ALLOCATED(atom->cc, false);
    ASSERT_ARRAY_ALLOCATED(atom->cc_flux, false);
    ASSERT_ARRAY_ALLOCATED(atom->edpd_temp, false);
    ASSERT_ARRAY_ALLOCATED(atom->edpd_flux, false);
    ASSERT_ARRAY_ALLOCATED(atom->edpd_cv, false);
    ASSERT_ARRAY_ALLOCATED(atom->length, false);
    ASSERT_ARRAY_ALLOCATED(atom->buckling, false);
    ASSERT_ARRAY_ALLOCATED(atom->bond_nt, false);
    ASSERT_ARRAY_ALLOCATED(atom->contact_radius, false);
    ASSERT_ARRAY_ALLOCATED(atom->smd_data_9, false);
    ASSERT_ARRAY_ALLOCATED(atom->smd_stress, false);
    ASSERT_ARRAY_ALLOCATED(atom->eff_plastic_strain, false);
    ASSERT_ARRAY_ALLOCATED(atom->eff_plastic_strain_rate, false);
    ASSERT_ARRAY_ALLOCATED(atom->damage, false);
    ASSERT_ARRAY_ALLOCATED(atom->rho, false);
    ASSERT_ARRAY_ALLOCATED(atom->drho, false);
    ASSERT_ARRAY_ALLOCATED(atom->esph, false);
    ASSERT_ARRAY_ALLOCATED(atom->desph, false);
    ASSERT_ARRAY_ALLOCATED(atom->cv, false);
    ASSERT_ARRAY_ALLOCATED(atom->vest, false);

    ASSERT_EQ(atom->nmolecule, expected.nmolecule);

    ASSERT_ARRAY_ALLOCATED(atom->molecules, expected.nmolecule > 0);

    ASSERT_EQ(atom->nivector, expected.nivector);
    ASSERT_EQ(atom->ndvector, expected.ndvector);

    ASSERT_ARRAY_ALLOCATED(atom->ivname, expected.has_ivname);
    ASSERT_ARRAY_ALLOCATED(atom->dvname, expected.has_dvname);
    ASSERT_ARRAY_ALLOCATED(atom->ianame, expected.has_ianame);
    ASSERT_ARRAY_ALLOCATED(atom->daname, expected.has_daname);
    ASSERT_ARRAY_ALLOCATED(atom->mass, expected.has_mass);
    ASSERT_ARRAY_ALLOCATED(atom->mass_setflag, expected.has_mass_setflag);

    ASSERT_EQ(atom->nextra_grow, expected.nextra_grow);
    ASSERT_EQ(atom->nextra_restart, expected.nextra_restart);
    ASSERT_EQ(atom->nextra_border, expected.nextra_border);
    ASSERT_EQ(atom->nextra_grow_max, expected.nextra_grow_max);
    ASSERT_EQ(atom->nextra_restart_max, expected.nextra_restart_max);
    ASSERT_EQ(atom->nextra_border_max, expected.nextra_border_max);
    ASSERT_EQ(atom->nextra_store, expected.nextra_store);

    ASSERT_ARRAY_ALLOCATED(atom->extra_grow, expected.has_extra_grow);
    ASSERT_ARRAY_ALLOCATED(atom->extra_restart, expected.has_extra_restart);
    ASSERT_ARRAY_ALLOCATED(atom->extra_border, expected.has_extra_border);
    ASSERT_ARRAY_ALLOCATED(atom->extra, expected.has_extra);
    ASSERT_ARRAY_ALLOCATED(atom->sametag, expected.has_sametag);

    ASSERT_EQ(atom->map_style, expected.map_style);
    ASSERT_EQ(atom->map_user, expected.map_user);
    ASSERT_EQ(atom->map_tag_max, expected.map_tag_max);
}

TEST_F(AtomStyleTest, atomic_is_default)
{
    AtomState expected;
    expected.atom_style = "atomic";
    expected.molecular  = Atom::ATOMIC;
    expected.tag_enable = 1;
    expected.has_type   = true;
    expected.has_mask   = true;
    expected.has_image  = true;
    expected.has_x      = true;
    expected.has_v      = true;
    expected.has_f      = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);
}

TEST_F(AtomStyleTest, atomic_after_charge)
{
    AtomState expected;
    expected.atom_style = "atomic";
    expected.molecular  = Atom::ATOMIC;
    expected.tag_enable = 1;
    expected.has_type   = true;
    expected.has_mask   = true;
    expected.has_image  = true;
    expected.has_x      = true;
    expected.has_v      = true;
    expected.has_f      = true;

    BEGIN_HIDE_OUTPUT();
    command("atom_style charge");
    command("atom_style atomic");
    END_HIDE_OUTPUT();

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);
}

TEST_F(AtomStyleTest, atomic)
{
    BEGIN_HIDE_OUTPUT();
    command("atom_modify map hash");
    command("create_box 2 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();

    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_HASH);
    ASSERT_EQ(lmp->atom->map_user, 2);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);
    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style atomic");
    command("pair_style zero 4.0");
    command("atom_modify map array");
    command("units real");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    auto x   = lmp->atom->x;
    auto v   = lmp->atom->v;
    auto tag = lmp->atom->tag;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);

    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 2);
    ASSERT_EQ(lmp->atom->nlocal, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);

    x = lmp->atom->x;
    v = lmp->atom->v;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);

    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 3);
    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->map_tag_max, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    BEGIN_HIDE_OUTPUT();
    command("comm_style tiled");
    command("change_box all triclinic");
    command("replicate 2 2 2");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->map_tag_max, 16);
    x   = lmp->atom->x;
    tag = lmp->atom->tag;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 6.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(7)][0], 6.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(7)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(7)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(8)][0], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(8)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(8)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(9)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(9)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(9)][2], 8.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][2], 7.9, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][0], 6.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][2], 8.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][0], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][2], 7.9, EPSILON);
    EXPECT_NEAR(x[GETIDX(13)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(13)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(13)][2], 8.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(14)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(14)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(14)][2], 7.9, EPSILON);
    EXPECT_NEAR(x[GETIDX(15)][0], 6.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(15)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(15)][2], 8.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(16)][0], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(16)][1], 10.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(16)][2], 7.9, EPSILON);
}

TEST_F(AtomStyleTest, charge)
{
    BEGIN_HIDE_OUTPUT();
    command("atom_style charge");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style = "charge";
    expected.molecular  = Atom::ATOMIC;
    expected.tag_enable = 1;
    expected.has_type   = true;
    expected.has_image  = true;
    expected.has_mask   = true;
    expected.has_x      = true;
    expected.has_v      = true;
    expected.has_f      = true;
    expected.q_flag     = 1;
    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 2 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("set atom 1 charge -0.5");
    command("set atom 2 charge  0.5");
    command("set atom 3 charge -1.0");
    command("set atom 4 charge  1.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);
    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style charge");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    auto x = lmp->atom->x;
    auto v = lmp->atom->v;
    auto q = lmp->atom->q;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], 0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(4)], 1.0, EPSILON);

    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    command("read_restart test_atom_styles.restart");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 2);
    ASSERT_EQ(lmp->atom->nlocal, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 3);

    x = lmp->atom->x;
    v = lmp->atom->v;
    q = lmp->atom->q;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -1.0, EPSILON);

    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    command("change_box all triclinic");
    command("replicate 2 2 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->map_tag_max, 16);
    q = lmp->atom->q;
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(4)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(5)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(6)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(7)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(8)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(9)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(10)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(11)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(12)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(13)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(14)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(15)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(16)], -1.0, EPSILON);
}

TEST_F(AtomStyleTest, sphere)
{
    BEGIN_HIDE_OUTPUT();
    command("atom_style sphere");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style  = "sphere";
    expected.molecular   = Atom::ATOMIC;
    expected.tag_enable  = 1;
    expected.sphere_flag = 1;
    expected.rmass_flag  = 1;
    expected.radius_flag = 1;
    expected.omega_flag  = 1;
    expected.torque_flag = 1;
    expected.has_type    = true;
    expected.has_mask    = true;
    expected.has_image   = true;
    expected.has_x       = true;
    expected.has_v       = true;
    expected.has_f       = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 2 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("set atom 1 mass 4.0");
    command("set atom 2 mass 4.0");
    command("set atom 3 mass 2.4");
    command("set atom 4 mass 2.4");
    command("set atom 1 omega -0.5  0.1  0.1");
    command("set atom 2 omega  0.5 -0.1 -0.1");
    command("set atom 3 omega -1.0  0.0  0.0");
    command("set atom 4 omega  0.0  1.0  0.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style sphere");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 1);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto rmass = lmp->atom->rmass;
    auto omega = lmp->atom->omega;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(1)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][0], -0.5, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][1], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][0], 0.5, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][1], -0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][0], -1.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][1], 1.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][2], 0.0, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2");
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    rmass = lmp->atom->rmass;
    omega = lmp->atom->omega;
    EXPECT_NEAR(rmass[GETIDX(1)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][0], -0.5, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][1], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][0], -1.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][0], -0.5, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][1], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(3)][2], 0.1, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][0], -1.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(omega[GETIDX(4)][2], 0.0, EPSILON);
}

TEST_F(AtomStyleTest, ellipsoid)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style ellipsoid");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style     = "ellipsoid";
    expected.molecular      = Atom::ATOMIC;
    expected.tag_enable     = 1;
    expected.ellipsoid_flag = 1;
    expected.rmass_flag     = 1;
    expected.angmom_flag    = 1;
    expected.torque_flag    = 1;
    expected.has_type       = true;
    expected.has_mask       = true;
    expected.has_image      = true;
    expected.has_x          = true;
    expected.has_v          = true;
    expected.has_f          = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("set type 1 mass 4.0");
    command("set type 2 mass 2.4");
    command("set type 3 mass 4.4");
    command("set type 1 shape 1.0 1.0 1.0");
    command("set type 2 shape 3.0 0.8 1.1");
    command("set atom 1 quat 1.0 0.0 0.0  0.0");
    command("set atom 2 quat 0.0 1.0 0.0 90.0");
    command("set atom 3 quat 1.0 0.0 1.0 30.0");
    command("set atom 4 quat 1.0 1.0 1.0 60.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style ellipsoid");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x         = lmp->atom->x;
    auto v         = lmp->atom->v;
    auto type      = lmp->atom->type;
    auto ellipsoid = lmp->atom->ellipsoid;
    auto rmass     = lmp->atom->rmass;
    auto avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    auto bonus     = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(4)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("comm_style tiled");
    command("replicate 1 1 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(9)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(11)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(12)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(9)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(11)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(12)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.25056280708573159, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(4)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(8)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(8)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.25056280708573159, EPSILON);
}

TEST_F(AtomStyleTest, line)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("dimension 2");
    command("atom_style line");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "line";
    expected.molecular     = Atom::ATOMIC;
    expected.tag_enable    = 1;
    expected.sphere_flag   = 1;
    expected.molecule_flag = 1;
    expected.line_flag     = 1;
    expected.rmass_flag    = 1;
    expected.radius_flag   = 1;
    expected.omega_flag    = 1;
    expected.torque_flag   = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box");
    command("create_atoms 1 single -2.0  2.0  0.0");
    command("create_atoms 1 single -2.0 -2.0  0.0");
    command("create_atoms 2 single  2.0  2.0  0.0");
    command("create_atoms 2 single  2.0 -2.0  0.0");
    command("create_atoms 3 single  3.0  0.0  0.0");
    command("create_atoms 3 single  0.0 -3.0  0.0");
    command("set type 1 mass 4.0");
    command("set type 2 mass 2.4");
    command("set type 3 mass 4.4");
    command("set type 1 length 2.0");
    command("set type 2 length 3.0");
    command("set atom 1 theta 0.0");
    command("set atom 2 theta 90.0");
    command("set atom 3 theta 30.0");
    command("set atom 4 theta 60.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("dimension 2");
    command("atom_style line");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->line_flag, 1);
    ASSERT_NE(lmp->atom->line, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto type  = lmp->atom->type;
    auto line  = lmp->atom->line;
    auto rmass = lmp->atom->rmass;
    auto avec  = (AtomVecLine *)lmp->atom->avec;
    auto bonus = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 3.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -3.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(2)], 1);
    ASSERT_EQ(line[GETIDX(3)], 2);
    ASSERT_EQ(line[GETIDX(4)], 3);
    ASSERT_EQ(line[GETIDX(5)], -1);
    ASSERT_EQ(line[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].theta, 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[1].theta, MathConst::MY_PI * 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[2].theta, MathConst::MY_PI / 6.0, EPSILON);
    EXPECT_NEAR(bonus[3].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[3].theta, MathConst::MY_PI / 3.0, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("comm_style tiled");
    command("change_box all triclinic");
    command("replicate 1 2 1 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type  = lmp->atom->type;
    line  = lmp->atom->line;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecLine *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(3)], 1);
    ASSERT_EQ(line[GETIDX(5)], -1);
    ASSERT_EQ(line[GETIDX(6)], -1);
    ASSERT_EQ(line[GETIDX(7)], 2);
    ASSERT_EQ(line[GETIDX(9)], 3);
    ASSERT_EQ(line[GETIDX(11)], -1);
    ASSERT_EQ(line[GETIDX(12)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(9)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(11)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(12)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].theta, 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[1].theta, MathConst::MY_PI / 6.0, EPSILON);
    EXPECT_NEAR(bonus[2].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[2].theta, 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[3].theta, MathConst::MY_PI / 6.0, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    line  = lmp->atom->line;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecLine *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(2)], -1);
    ASSERT_EQ(line[GETIDX(3)], 1);
    ASSERT_EQ(line[GETIDX(4)], -1);
    ASSERT_EQ(line[GETIDX(5)], 2);
    ASSERT_EQ(line[GETIDX(6)], -1);
    ASSERT_EQ(line[GETIDX(7)], 3);
    ASSERT_EQ(line[GETIDX(8)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(8)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].theta, 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[1].theta, MathConst::MY_PI / 6.0, EPSILON);
    EXPECT_NEAR(bonus[2].length, 2.0, EPSILON);
    EXPECT_NEAR(bonus[2].theta, 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].length, 3.0, EPSILON);
    EXPECT_NEAR(bonus[3].theta, MathConst::MY_PI / 6.0, EPSILON);
}

TEST_F(AtomStyleTest, tri)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style tri");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "tri";
    expected.molecular     = Atom::ATOMIC;
    expected.tag_enable    = 1;
    expected.sphere_flag   = 1;
    expected.molecule_flag = 1;
    expected.tri_flag      = 1;
    expected.rmass_flag    = 1;
    expected.radius_flag   = 1;
    expected.omega_flag    = 1;
    expected.angmom_flag   = 1;
    expected.torque_flag   = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("set type 1 mass 4.0");
    command("set type 2 mass 2.4");
    command("set type 3 mass 4.4");
    command("set type 1 tri 1.0");
    command("set type 2 tri 1.5");
    command("set atom 1 quat 1.0 0.0 0.0  0.0");
    command("set atom 2 quat 0.0 1.0 0.0 90.0");
    command("set atom 3 quat 1.0 0.0 1.0 30.0");
    command("set atom 4 quat 1.0 1.0 1.0 60.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style tri");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tri_flag, 1);
    ASSERT_NE(lmp->atom->tri, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x      = lmp->atom->x;
    auto v      = lmp->atom->v;
    auto type   = lmp->atom->type;
    auto tri    = lmp->atom->tri;
    auto rmass  = lmp->atom->rmass;
    auto radius = lmp->atom->radius;
    auto avec   = (AtomVecTri *)lmp->atom->avec;
    auto bonus  = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(2)], 1);
    ASSERT_EQ(tri[GETIDX(3)], 2);
    ASSERT_EQ(tri[GETIDX(4)], 3);
    ASSERT_EQ(tri[GETIDX(5)], -1);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(radius[GETIDX(1)], 0.5773502691896258, EPSILON);
    EXPECT_NEAR(radius[GETIDX(2)], 0.5773502691896258, EPSILON);
    EXPECT_NEAR(radius[GETIDX(3)], 0.8660254037844390, EPSILON);
    EXPECT_NEAR(radius[GETIDX(4)], 0.8660254037844390, EPSILON);
    EXPECT_NEAR(radius[GETIDX(5)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(6)], 0.5, EPSILON);

    EXPECT_NEAR(bonus[0].inertia[0], 14.017974903242481, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 13.94589575227541, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.072258416330334363, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 13.982119044342252, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 13.945895752275419, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.10811427523057447, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 19.15175691481879, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 18.948744087979005, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.23541253382609079, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 19.018309360029388, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 18.948744087979012, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.36886008861549813, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 0.66466395261228639, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], -0.26579965871355399, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], -0.64169714094040209, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], -0.27531282359251713, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.63718542087921404, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.66984067651944412, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], -0.26272786480888066, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], -0.27619246288035992, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.70865471105868871, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.40297854431552654, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], -0.56684324752832238, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.11876668744732566, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.55997606330452898, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.69182747953492685, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.072026021657128514, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.45012642589672475, EPSILON);
    EXPECT_NEAR(bonus[0].c1[0], -0.0052525338293288905, EPSILON);
    EXPECT_NEAR(bonus[0].c1[1], -0.55767753582520529, EPSILON);
    EXPECT_NEAR(bonus[0].c1[2], 0.14933690186163631, EPSILON);
    EXPECT_NEAR(bonus[1].c1[0], 0.18561232929671426, EPSILON);
    EXPECT_NEAR(bonus[1].c1[1], 0.51379191773154387, EPSILON);
    EXPECT_NEAR(bonus[1].c1[2], 0.18681344121910506, EPSILON);
    EXPECT_NEAR(bonus[2].c1[0], -0.22123552085772166, EPSILON);
    EXPECT_NEAR(bonus[2].c1[1], -0.28372571379613187, EPSILON);
    EXPECT_NEAR(bonus[2].c1[2], 0.78775285695558617, EPSILON);
    EXPECT_NEAR(bonus[3].c1[0], -0.69845793336676587, EPSILON);
    EXPECT_NEAR(bonus[3].c1[1], 0.18083523090249506, EPSILON);
    EXPECT_NEAR(bonus[3].c1[2], 0.47901475403318056, EPSILON);
    EXPECT_NEAR(bonus[0].c2[0], 0.019602723119529659, EPSILON);
    EXPECT_NEAR(bonus[0].c2[1], 0.14942924536134222, EPSILON);
    EXPECT_NEAR(bonus[0].c2[2], -0.55733290519255385, EPSILON);
    EXPECT_NEAR(bonus[1].c2[0], 0.2208094914375279, EPSILON);
    EXPECT_NEAR(bonus[1].c2[1], -0.4849604211463005, EPSILON);
    EXPECT_NEAR(bonus[1].c2[2], 0.22223836695322477, EPSILON);
    EXPECT_NEAR(bonus[2].c2[0], 0.17705460333259249, EPSILON);
    EXPECT_NEAR(bonus[2].c2[1], -0.56674478453558153, EPSILON);
    EXPECT_NEAR(bonus[2].c2[2], -0.6304379562181005, EPSILON);
    EXPECT_NEAR(bonus[3].c2[0], 0.22007613459534958, EPSILON);
    EXPECT_NEAR(bonus[3].c2[1], -0.82388470022624394, EPSILON);
    EXPECT_NEAR(bonus[3].c2[2], -0.15093208974463557, EPSILON);
    EXPECT_NEAR(bonus[0].c3[0], -0.014350189290200811, EPSILON);
    EXPECT_NEAR(bonus[0].c3[1], 0.40824829046386302, EPSILON);
    EXPECT_NEAR(bonus[0].c3[2], 0.40799600333091751, EPSILON);
    EXPECT_NEAR(bonus[1].c3[0], -0.40642182073424188, EPSILON);
    EXPECT_NEAR(bonus[1].c3[1], -0.028831496585242929, EPSILON);
    EXPECT_NEAR(bonus[1].c3[2], -0.40905180817232945, EPSILON);
    EXPECT_NEAR(bonus[2].c3[0], 0.044180917525129149, EPSILON);
    EXPECT_NEAR(bonus[2].c3[1], 0.85047049833171351, EPSILON);
    EXPECT_NEAR(bonus[2].c3[2], -0.15731490073748589, EPSILON);
    EXPECT_NEAR(bonus[3].c3[0], 0.47838179877141634, EPSILON);
    EXPECT_NEAR(bonus[3].c3[1], 0.64304946932374796, EPSILON);
    EXPECT_NEAR(bonus[3].c3[2], -0.32808266428854477, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("change_box all triclinic");
    command("replicate 1 1 2");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type   = lmp->atom->type;
    tri    = lmp->atom->tri;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    avec   = (AtomVecTri *)lmp->atom->avec;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(3)], 1);
    ASSERT_EQ(tri[GETIDX(5)], -1);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    ASSERT_EQ(tri[GETIDX(7)], 2);
    ASSERT_EQ(tri[GETIDX(9)], 3);
    ASSERT_EQ(tri[GETIDX(11)], -1);
    ASSERT_EQ(tri[GETIDX(12)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(9)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(11)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(12)], 4.4, EPSILON);
    EXPECT_NEAR(radius[GETIDX(1)], 0.5773502691896258, EPSILON);
    EXPECT_NEAR(radius[GETIDX(3)], 0.8660254037844390, EPSILON);
    EXPECT_NEAR(radius[GETIDX(5)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(6)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(7)], 0.5773502691896258, EPSILON);
    EXPECT_NEAR(radius[GETIDX(9)], 0.8660254037844390, EPSILON);
    EXPECT_NEAR(radius[GETIDX(11)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(12)], 0.5, EPSILON);

    EXPECT_NEAR(bonus[0].inertia[0], 14.017974903242481, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 13.94589575227541, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.072258416330334363, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 19.15175691481879, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 18.948744087979005, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.23541253382609079, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 14.017974903242481, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 13.94589575227541, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.072258416330334363, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 19.15175691481879, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 18.948744087979005, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.23541253382609079, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 0.66466395261228639, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], -0.26579965871355399, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], -0.64169714094040209, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], -0.27531282359251713, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.70865471105868871, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.40297854431552654, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], -0.56684324752832238, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.11876668744732566, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.66466395261228639, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], -0.26579965871355399, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], -0.64169714094040209, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], -0.27531282359251713, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.70865471105868871, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.40297854431552654, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], -0.56684324752832238, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.11876668744732566, EPSILON);
    EXPECT_NEAR(bonus[0].c1[0], -0.0052525338293288879, EPSILON);
    EXPECT_NEAR(bonus[0].c1[1], -0.55767753582520529, EPSILON);
    EXPECT_NEAR(bonus[0].c1[2], 0.14933690186163626, EPSILON);
    EXPECT_NEAR(bonus[1].c1[0], -0.22123552085772158, EPSILON);
    EXPECT_NEAR(bonus[1].c1[1], -0.28372571379613187, EPSILON);
    EXPECT_NEAR(bonus[1].c1[2], 0.78775285695558628, EPSILON);
    EXPECT_NEAR(bonus[2].c1[0], -0.0052525338293288905, EPSILON);
    EXPECT_NEAR(bonus[2].c1[1], -0.55767753582520529, EPSILON);
    EXPECT_NEAR(bonus[2].c1[2], 0.14933690186163631, EPSILON);
    EXPECT_NEAR(bonus[3].c1[0], -0.22123552085772166, EPSILON);
    EXPECT_NEAR(bonus[3].c1[1], -0.28372571379613187, EPSILON);
    EXPECT_NEAR(bonus[3].c1[2], 0.78775285695558617, EPSILON);
    EXPECT_NEAR(bonus[0].c2[0], 0.019602723119529659, EPSILON);
    EXPECT_NEAR(bonus[0].c2[1], 0.14942924536134222, EPSILON);
    EXPECT_NEAR(bonus[0].c2[2], -0.55733290519255385, EPSILON);
    EXPECT_NEAR(bonus[1].c2[0], 0.17705460333259249, EPSILON);
    EXPECT_NEAR(bonus[1].c2[1], -0.56674478453558153, EPSILON);
    EXPECT_NEAR(bonus[1].c2[2], -0.6304379562181005, EPSILON);
    EXPECT_NEAR(bonus[2].c2[0], 0.019602723119529659, EPSILON);
    EXPECT_NEAR(bonus[2].c2[1], 0.14942924536134222, EPSILON);
    EXPECT_NEAR(bonus[2].c2[2], -0.55733290519255385, EPSILON);
    EXPECT_NEAR(bonus[3].c2[0], 0.17705460333259249, EPSILON);
    EXPECT_NEAR(bonus[3].c2[1], -0.56674478453558153, EPSILON);
    EXPECT_NEAR(bonus[3].c2[2], -0.6304379562181005, EPSILON);
    EXPECT_NEAR(bonus[0].c3[0], -0.014350189290200811, EPSILON);
    EXPECT_NEAR(bonus[0].c3[1], 0.40824829046386302, EPSILON);
    EXPECT_NEAR(bonus[0].c3[2], 0.40799600333091751, EPSILON);
    EXPECT_NEAR(bonus[1].c3[0], 0.044180917525129149, EPSILON);
    EXPECT_NEAR(bonus[1].c3[1], 0.85047049833171351, EPSILON);
    EXPECT_NEAR(bonus[1].c3[2], -0.15731490073748589, EPSILON);
    EXPECT_NEAR(bonus[2].c3[0], -0.014350189290200811, EPSILON);
    EXPECT_NEAR(bonus[2].c3[1], 0.40824829046386302, EPSILON);
    EXPECT_NEAR(bonus[2].c3[2], 0.40799600333091751, EPSILON);
    EXPECT_NEAR(bonus[3].c3[0], 0.044180917525129149, EPSILON);
    EXPECT_NEAR(bonus[3].c3[1], 0.85047049833171351, EPSILON);
    EXPECT_NEAR(bonus[3].c3[2], -0.15731490073748589, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    tri   = lmp->atom->tri;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecTri *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(2)], -1);
    ASSERT_EQ(tri[GETIDX(3)], 1);
    ASSERT_EQ(tri[GETIDX(4)], -1);
    ASSERT_EQ(tri[GETIDX(5)], 2);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    ASSERT_EQ(tri[GETIDX(7)], 3);
    ASSERT_EQ(tri[GETIDX(8)], -1);
    EXPECT_NEAR(bonus[0].inertia[0], 14.017974903242481, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 13.94589575227541, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.072258416330334363, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 19.15175691481879, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 18.948744087979005, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.23541253382609079, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 14.017974903242481, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 13.94589575227541, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.072258416330334363, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 19.15175691481879, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 18.948744087979005, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.23541253382609079, EPSILON);
}

TEST_F(AtomStyleTest, body_nparticle)
{
    if (!LAMMPS::is_installed_pkg("BODY")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style body nparticle 2 4");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style  = "body";
    expected.molecular   = Atom::ATOMIC;
    expected.tag_enable  = 1;
    expected.body_flag   = 1;
    expected.rmass_flag  = 1;
    expected.radius_flag = 1;
    expected.angmom_flag = 1;
    expected.torque_flag = 1;
    expected.has_type    = true;
    expected.has_mask    = true;
    expected.has_image   = true;
    expected.has_x       = true;
    expected.has_v       = true;
    expected.has_f       = true;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    auto avec = (AtomVecBody *)lmp->atom->avec;
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_NE(avec->bptr, nullptr);
    ASSERT_THAT(std::string(avec->bptr->style), Eq("nparticle"));

    const char data_file[] = "\n4 atoms\n"
                             "4 bodies\n"
                             "3 atom types\n\n"
                             "-4.0 4.0 xlo xhi\n"
                             "-4.0 4.0 ylo yhi\n"
                             "-4.0 4.0 zlo zhi\n\n"
                             "Atoms # body\n\n"
                             "1 1 1 4.0 -2.0  2.0 0.1\n"
                             "2 1 1 4.0 -2.0 -2.0 -0.1\n"
                             "3 2 1 2.4  2.0  2.0 -0.1\n"
                             "4 2 1 2.4  2.0 -2.0 0.1\n\n"
                             "Bodies\n\n"
                             "1 1 12\n"
                             "2\n"
                             "2.0 2.0 0.0 0.0 0.0 0.0\n"
                             "-1.0 0.0 0.0\n"
                             "1.0 0.0 0.0\n"
                             "2 1 15\n"
                             "3\n"
                             "0.5 4.0 4.5 0.0 0.0 0.0\n"
                             "1.0 -0.5 0.0\n"
                             "1.0  0.5 0.0\n"
                             "-1.0 0.0 0.0\n"
                             "3 1 18\n"
                             "4\n"
                             "1.67188 1.46875 0.796875 0 0 0.546875\n"
                             "0.0  0.0 1.0\n"
                             "0.75 0.0 -0.25\n"
                             "0.5  0.5 -0.25\n"
                             "0.5 -0.5 -0.25\n"
                             "4 1 12\n"
                             "2\n"
                             "12.0 0.0 12.0 0.0 0.0 0.0\n"
                             "0.0 1.0 0.0\n"
                             "0.0 -3.0 0.0\n";
    FILE *fp = fopen("input_atom_styles.data", "w");
    fputs(data_file, fp);
    fclose(fp);
    BEGIN_HIDE_OUTPUT();
    command("atom_modify map array");
    command("read_data input_atom_styles.data");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("set type 3 mass 4.4");
    command("set atom 1 quat 1.0 0.0 0.0  0.0");
    command("set atom 2 quat 0.0 1.0 0.0 90.0");
    command("set atom 3 quat 1.0 0.0 1.0 30.0");
    command("set atom 4 quat 1.0 1.0 1.0 60.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    auto x      = lmp->atom->x;
    auto v      = lmp->atom->v;
    auto type   = lmp->atom->type;
    auto body   = lmp->atom->body;
    auto rmass  = lmp->atom->rmass;
    auto radius = lmp->atom->radius;
    auto angmom = lmp->atom->angmom;
    auto bonus  = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    EXPECT_NEAR(angmom[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][2], 0.0, EPSILON);

    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], 1);
    ASSERT_EQ(body[GETIDX(3)], 2);
    ASSERT_EQ(body[GETIDX(4)], 3);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(radius[GETIDX(1)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(2)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(3)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(4)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(5)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(6)], 0.5, EPSILON);

    EXPECT_NEAR(bonus[0].inertia[0], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 4.5, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 4.0, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 1.7746273249544022, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 1.67188, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.49099767504559777, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 12.0, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 12.0, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), EPSILON);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 1);
    ASSERT_EQ(bonus[2].ilocal, 2);
    ASSERT_EQ(bonus[3].ilocal, 3);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 9);
    ASSERT_EQ(bonus[2].ndouble, 12);
    ASSERT_EQ(bonus[3].ndouble, 6);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 3);
    ASSERT_EQ(bonus[2].ivalue[0], 4);
    ASSERT_EQ(bonus[3].ivalue[0], 2);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style body nparticle 2 4");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->body_flag, 1);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    x      = lmp->atom->x;
    v      = lmp->atom->v;
    type   = lmp->atom->type;
    body   = lmp->atom->body;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    angmom = lmp->atom->angmom;
    avec   = (AtomVecBody *)lmp->atom->avec;
    bonus  = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    EXPECT_NEAR(angmom[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(angmom[GETIDX(4)][2], 0.0, EPSILON);

    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], 1);
    ASSERT_EQ(body[GETIDX(3)], 2);
    ASSERT_EQ(body[GETIDX(4)], 3);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(radius[GETIDX(1)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(2)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(3)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(4)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(5)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(6)], 0.5, EPSILON);

    EXPECT_NEAR(bonus[0].inertia[0], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 4.5, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 4.0, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 1.7746273249544022, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 1.67188, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.49099767504559777, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 12.0, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 12.0, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], -sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.25056280708573148, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.62499650256800654, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.47323774316465234, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.33072552332373728, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.52540083597613996, EPSILON);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 1);
    ASSERT_EQ(bonus[2].ilocal, 2);
    ASSERT_EQ(bonus[3].ilocal, 3);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 9);
    ASSERT_EQ(bonus[2].ndouble, 12);
    ASSERT_EQ(bonus[3].ndouble, 6);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 3);
    ASSERT_EQ(bonus[2].ivalue[0], 4);
    ASSERT_EQ(bonus[3].ivalue[0], 2);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("comm_style tiled");
    command("replicate 1 1 2");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    avec = (AtomVecBody *)lmp->atom->avec;
    ASSERT_THAT(std::string(avec->bptr->style), Eq("nparticle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type   = lmp->atom->type;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    body   = lmp->atom->body;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(3)], 1);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_EQ(body[GETIDX(7)], 2);
    ASSERT_EQ(body[GETIDX(9)], 3);
    ASSERT_EQ(body[GETIDX(11)], -1);
    ASSERT_EQ(body[GETIDX(12)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(9)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(11)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(12)], 4.4, EPSILON);
    EXPECT_NEAR(radius[GETIDX(1)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(3)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(5)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(6)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(7)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(9)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(11)], 0.5, EPSILON);
    EXPECT_NEAR(radius[GETIDX(12)], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[0], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[1], 2.0, EPSILON);
    EXPECT_NEAR(bonus[0].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[0], 1.7746273249544022, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[1], 1.67188, EPSILON);
    EXPECT_NEAR(bonus[1].inertia[2], 0.49099767504559777, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[0], 2.0, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[1], 2.0, EPSILON);
    EXPECT_NEAR(bonus[2].inertia[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[0], 1.7746273249544022, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[1], 1.67188, EPSILON);
    EXPECT_NEAR(bonus[3].inertia[2], 0.49099767504559777, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.25056280708573148, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.25056280708573148, EPSILON);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 2);
    ASSERT_EQ(bonus[2].ilocal, 4);
    ASSERT_EQ(bonus[3].ilocal, 6);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 12);
    ASSERT_EQ(bonus[2].ndouble, 6);
    ASSERT_EQ(bonus[3].ndouble, 12);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 4);
    ASSERT_EQ(bonus[2].ivalue[0], 2);
    ASSERT_EQ(bonus[3].ivalue[0], 4);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    body   = lmp->atom->body;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    avec   = (AtomVecBody *)lmp->atom->avec;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], -1);
    ASSERT_EQ(body[GETIDX(3)], 1);
    ASSERT_EQ(body[GETIDX(4)], -1);
    ASSERT_EQ(body[GETIDX(5)], 2);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_EQ(body[GETIDX(7)], 3);
    ASSERT_EQ(body[GETIDX(8)], -1);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 2);
    ASSERT_EQ(bonus[2].ilocal, 4);
    ASSERT_EQ(bonus[3].ilocal, 6);
}

TEST_F(AtomStyleTest, template)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();
    BEGIN_HIDE_OUTPUT();
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style template twomols");
    command("newton on");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "template";
    expected.molecular     = Atom::TEMPLATE;
    expected.nbondtypes    = 2;
    expected.nangletypes   = 2;
    expected.tag_enable    = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;
    expected.molecule_flag = 1;
    expected.molindex_flag = 1;
    expected.molatom_flag  = 1;
    expected.nmolecule     = 2;
    expected.map_style     = 3;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 4 box bond/types 2 angle/types 2 ");
    command("create_atoms 0 single -2.0  2.0  0.1 mol twomols 65234");
    command("create_atoms 0 single -2.0 -2.0 -0.1 mol twomols 62346");
    command("create_atoms 0 single  2.0  2.0 -0.1 mol twomols 61354");
    command("create_atoms 3 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 4 single  2.0 -2.0  2.1");
    command("mass 1 16.0");
    command("mass 2 1.0");
    command("mass 3 12.0");
    command("mass 4 16.0");
    command("bond_style zero");
    command("bond_coeff 1 1.0");
    command("bond_coeff 2 1.16");
    command("angle_style zero");
    command("angle_coeff * 109.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("template"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data");
    command("clear");
    command("units real");
    command("newton off on");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style template twomols");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("angle_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("template"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    auto molecule = lmp->atom->molecule;
    auto molindex = lmp->atom->molindex;
    auto molatom  = lmp->atom->molatom;

    ASSERT_EQ(molecule[GETIDX(1)], 1);
    ASSERT_EQ(molecule[GETIDX(2)], 1);
    ASSERT_EQ(molecule[GETIDX(3)], 1);
    ASSERT_EQ(molecule[GETIDX(4)], 2);
    ASSERT_EQ(molecule[GETIDX(5)], 2);
    ASSERT_EQ(molecule[GETIDX(6)], 2);
    ASSERT_EQ(molecule[GETIDX(7)], 3);
    ASSERT_EQ(molecule[GETIDX(8)], 3);
    ASSERT_EQ(molecule[GETIDX(9)], 3);
    ASSERT_EQ(molecule[GETIDX(10)], 0);
    ASSERT_EQ(molecule[GETIDX(11)], 0);
    ASSERT_EQ(molecule[GETIDX(12)], 0);
    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(7)], 0);
    ASSERT_EQ(molindex[GETIDX(8)], 0);
    ASSERT_EQ(molindex[GETIDX(9)], 0);
    ASSERT_EQ(molindex[GETIDX(10)], -1);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], 0);
    ASSERT_EQ(molatom[GETIDX(8)], 1);
    ASSERT_EQ(molatom[GETIDX(9)], 2);
    ASSERT_EQ(molatom[GETIDX(10)], -1);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);

    BEGIN_HIDE_OUTPUT();
    command("clear");
    command("units real");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style template twomols");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("angle_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("template"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;

    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(7)], 0);
    ASSERT_EQ(molindex[GETIDX(8)], 0);
    ASSERT_EQ(molindex[GETIDX(9)], 0);
    ASSERT_EQ(molindex[GETIDX(10)], -1);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], 0);
    ASSERT_EQ(molatom[GETIDX(8)], 1);
    ASSERT_EQ(molatom[GETIDX(9)], 2);
    ASSERT_EQ(molatom[GETIDX(10)], -1);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);

    auto x    = lmp->atom->x;
    auto v    = lmp->atom->v;
    auto type = lmp->atom->type;

    EXPECT_NEAR(x[GETIDX(10)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(8)], 2);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(10)], 3);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 4);

    BEGIN_HIDE_OUTPUT();
    command("group two id 7:10");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("template"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 16);
    ASSERT_EQ(lmp->atom->nbonds, 8);
    ASSERT_EQ(lmp->atom->nangles, 4);
    ASSERT_EQ(lmp->atom->nlocal, 16);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 24);

    type     = lmp->atom->type;
    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 4);
    ASSERT_EQ(type[GETIDX(13)], 1);
    ASSERT_EQ(type[GETIDX(14)], 2);
    ASSERT_EQ(type[GETIDX(15)], 2);
    ASSERT_EQ(type[GETIDX(16)], 1);
    ASSERT_EQ(type[GETIDX(17)], 2);
    ASSERT_EQ(type[GETIDX(18)], 2);
    ASSERT_EQ(type[GETIDX(23)], 3);
    ASSERT_EQ(type[GETIDX(24)], 4);

    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molindex[GETIDX(13)], 0);
    ASSERT_EQ(molindex[GETIDX(14)], 0);
    ASSERT_EQ(molindex[GETIDX(15)], 0);
    ASSERT_EQ(molindex[GETIDX(16)], 0);
    ASSERT_EQ(molindex[GETIDX(17)], 0);
    ASSERT_EQ(molindex[GETIDX(18)], 0);
    ASSERT_EQ(molindex[GETIDX(23)], -1);
    ASSERT_EQ(molindex[GETIDX(24)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(13)], 0);
    ASSERT_EQ(molatom[GETIDX(14)], 1);
    ASSERT_EQ(molatom[GETIDX(15)], 2);
    ASSERT_EQ(molatom[GETIDX(16)], 0);
    ASSERT_EQ(molatom[GETIDX(17)], 1);
    ASSERT_EQ(molatom[GETIDX(18)], 2);
    ASSERT_EQ(molatom[GETIDX(23)], -1);
    ASSERT_EQ(molatom[GETIDX(24)], -1);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 16);

    type     = lmp->atom->type;
    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(7)], 4);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(type[GETIDX(9)], 1);
    ASSERT_EQ(type[GETIDX(10)], 2);
    ASSERT_EQ(type[GETIDX(11)], 2);
    ASSERT_EQ(type[GETIDX(12)], 1);
    ASSERT_EQ(type[GETIDX(13)], 2);
    ASSERT_EQ(type[GETIDX(14)], 2);
    ASSERT_EQ(type[GETIDX(15)], 4);
    ASSERT_EQ(type[GETIDX(16)], 3);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], -1);
    ASSERT_EQ(molatom[GETIDX(8)], -1);
    ASSERT_EQ(molatom[GETIDX(9)], 0);
    ASSERT_EQ(molatom[GETIDX(10)], 1);
    ASSERT_EQ(molatom[GETIDX(11)], 2);
    ASSERT_EQ(molatom[GETIDX(12)], 0);
    ASSERT_EQ(molatom[GETIDX(13)], 1);
    ASSERT_EQ(molatom[GETIDX(14)], 2);
    ASSERT_EQ(molatom[GETIDX(15)], -1);
    ASSERT_EQ(molatom[GETIDX(16)], -1);
}

TEST_F(AtomStyleTest, template_charge)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();
    BEGIN_HIDE_OUTPUT();
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style hybrid template twomols charge");
    command("newton on");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "hybrid";
    expected.molecular     = Atom::TEMPLATE;
    expected.nbondtypes    = 2;
    expected.nangletypes   = 2;
    expected.tag_enable    = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;
    expected.molecule_flag = 1;
    expected.molindex_flag = 1;
    expected.molatom_flag  = 1;
    expected.q_flag        = 1;
    expected.nmolecule     = 2;
    expected.map_style     = 3;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    auto hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("template"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("charge"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);

    BEGIN_HIDE_OUTPUT();
    command("create_box 4 box bond/types 2 angle/types 2 ");
    command("create_atoms 0 single -2.0  2.0  0.1 mol twomols 65234");
    command("create_atoms 0 single -2.0 -2.0 -0.1 mol twomols 62346");
    command("create_atoms 0 single  2.0  2.0 -0.1 mol twomols 61354");
    command("create_atoms 3 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 4 single  2.0 -2.0  2.1");
    command("mass 1 16.0");
    command("mass 2 1.0");
    command("mass 3 12.0");
    command("mass 4 16.0");
    command("set atom 10 charge 0.7");
    command("set atom 11 charge -0.35");
    command("set atom 12 charge -0.35");
    command("bond_style zero");
    command("bond_coeff 1 1.0");
    command("bond_coeff 2 1.16");
    command("angle_style zero");
    command("angle_coeff * 109.0");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_NE(lmp->atom->avec, nullptr);
    hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("template"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("charge"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);

    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data");
    command("clear");
    command("units real");
    command("newton off on");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style hybrid template twomols charge");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("angle_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    auto molecule = lmp->atom->molecule;
    auto molindex = lmp->atom->molindex;
    auto molatom  = lmp->atom->molatom;

    ASSERT_EQ(molecule[GETIDX(1)], 1);
    ASSERT_EQ(molecule[GETIDX(2)], 1);
    ASSERT_EQ(molecule[GETIDX(3)], 1);
    ASSERT_EQ(molecule[GETIDX(4)], 2);
    ASSERT_EQ(molecule[GETIDX(5)], 2);
    ASSERT_EQ(molecule[GETIDX(6)], 2);
    ASSERT_EQ(molecule[GETIDX(7)], 3);
    ASSERT_EQ(molecule[GETIDX(8)], 3);
    ASSERT_EQ(molecule[GETIDX(9)], 3);
    ASSERT_EQ(molecule[GETIDX(10)], 0);
    ASSERT_EQ(molecule[GETIDX(11)], 0);
    ASSERT_EQ(molecule[GETIDX(12)], 0);
    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(7)], 0);
    ASSERT_EQ(molindex[GETIDX(8)], 0);
    ASSERT_EQ(molindex[GETIDX(9)], 0);
    ASSERT_EQ(molindex[GETIDX(10)], -1);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], 0);
    ASSERT_EQ(molatom[GETIDX(8)], 1);
    ASSERT_EQ(molatom[GETIDX(9)], 2);
    ASSERT_EQ(molatom[GETIDX(10)], -1);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);

    BEGIN_HIDE_OUTPUT();
    command("clear");
    command("units real");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("atom_style hybrid template twomols charge");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("angle_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 12);
    ASSERT_EQ(lmp->atom->nlocal, 12);
    ASSERT_EQ(lmp->atom->nbonds, 6);
    ASSERT_EQ(lmp->atom->nangles, 3);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;

    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(7)], 0);
    ASSERT_EQ(molindex[GETIDX(8)], 0);
    ASSERT_EQ(molindex[GETIDX(9)], 0);
    ASSERT_EQ(molindex[GETIDX(10)], -1);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], 0);
    ASSERT_EQ(molatom[GETIDX(8)], 1);
    ASSERT_EQ(molatom[GETIDX(9)], 2);
    ASSERT_EQ(molatom[GETIDX(10)], -1);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);

    auto x    = lmp->atom->x;
    auto v    = lmp->atom->v;
    auto type = lmp->atom->type;
    auto q    = lmp->atom->q;

    EXPECT_NEAR(x[GETIDX(10)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(10)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(11)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(12)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(7)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(8)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(9)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(10)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(11)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(12)][2], 0.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(1)], -0.8472, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(4)], -0.8472, EPSILON);
    EXPECT_NEAR(q[GETIDX(5)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(6)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(7)], -0.8472, EPSILON);
    EXPECT_NEAR(q[GETIDX(8)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(9)], 0.4236, EPSILON);
    EXPECT_NEAR(q[GETIDX(10)], 0.7, EPSILON);
    EXPECT_NEAR(q[GETIDX(11)], -0.35, EPSILON);
    EXPECT_NEAR(q[GETIDX(12)], -0.35, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(8)], 2);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(10)], 3);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 4);

    BEGIN_HIDE_OUTPUT();
    command("group two id 7:10");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("molecule twomols h2o.mol co2.mol offset 2 1 1 0 0");
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 16);

    ASSERT_EQ(lmp->atom->nlocal, 16);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::TEMPLATE);
    ASSERT_EQ(lmp->atom->ntypes, 4);
    ASSERT_EQ(lmp->atom->nbonds, 8);
    ASSERT_EQ(lmp->atom->nangles, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 24);

    type     = lmp->atom->type;
    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 4);
    ASSERT_EQ(type[GETIDX(13)], 1);
    ASSERT_EQ(type[GETIDX(14)], 2);
    ASSERT_EQ(type[GETIDX(15)], 2);
    ASSERT_EQ(type[GETIDX(16)], 1);
    ASSERT_EQ(type[GETIDX(17)], 2);
    ASSERT_EQ(type[GETIDX(18)], 2);
    ASSERT_EQ(type[GETIDX(23)], 3);
    ASSERT_EQ(type[GETIDX(24)], 4);

    ASSERT_EQ(molindex[GETIDX(1)], 0);
    ASSERT_EQ(molindex[GETIDX(2)], 0);
    ASSERT_EQ(molindex[GETIDX(3)], 0);
    ASSERT_EQ(molindex[GETIDX(4)], 0);
    ASSERT_EQ(molindex[GETIDX(5)], 0);
    ASSERT_EQ(molindex[GETIDX(6)], 0);
    ASSERT_EQ(molindex[GETIDX(11)], -1);
    ASSERT_EQ(molindex[GETIDX(12)], -1);
    ASSERT_EQ(molindex[GETIDX(13)], 0);
    ASSERT_EQ(molindex[GETIDX(14)], 0);
    ASSERT_EQ(molindex[GETIDX(15)], 0);
    ASSERT_EQ(molindex[GETIDX(16)], 0);
    ASSERT_EQ(molindex[GETIDX(17)], 0);
    ASSERT_EQ(molindex[GETIDX(18)], 0);
    ASSERT_EQ(molindex[GETIDX(23)], -1);
    ASSERT_EQ(molindex[GETIDX(24)], -1);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(11)], -1);
    ASSERT_EQ(molatom[GETIDX(12)], -1);
    ASSERT_EQ(molatom[GETIDX(13)], 0);
    ASSERT_EQ(molatom[GETIDX(14)], 1);
    ASSERT_EQ(molatom[GETIDX(15)], 2);
    ASSERT_EQ(molatom[GETIDX(16)], 0);
    ASSERT_EQ(molatom[GETIDX(17)], 1);
    ASSERT_EQ(molatom[GETIDX(18)], 2);
    ASSERT_EQ(molatom[GETIDX(23)], -1);
    ASSERT_EQ(molatom[GETIDX(24)], -1);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 16);

    type     = lmp->atom->type;
    molecule = lmp->atom->molecule;
    molindex = lmp->atom->molindex;
    molatom  = lmp->atom->molatom;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 2);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 1);
    ASSERT_EQ(type[GETIDX(5)], 2);
    ASSERT_EQ(type[GETIDX(6)], 2);
    ASSERT_EQ(type[GETIDX(7)], 4);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(type[GETIDX(9)], 1);
    ASSERT_EQ(type[GETIDX(10)], 2);
    ASSERT_EQ(type[GETIDX(11)], 2);
    ASSERT_EQ(type[GETIDX(12)], 1);
    ASSERT_EQ(type[GETIDX(13)], 2);
    ASSERT_EQ(type[GETIDX(14)], 2);
    ASSERT_EQ(type[GETIDX(15)], 4);
    ASSERT_EQ(type[GETIDX(16)], 3);
    ASSERT_EQ(molatom[GETIDX(1)], 0);
    ASSERT_EQ(molatom[GETIDX(2)], 1);
    ASSERT_EQ(molatom[GETIDX(3)], 2);
    ASSERT_EQ(molatom[GETIDX(4)], 0);
    ASSERT_EQ(molatom[GETIDX(5)], 1);
    ASSERT_EQ(molatom[GETIDX(6)], 2);
    ASSERT_EQ(molatom[GETIDX(7)], -1);
    ASSERT_EQ(molatom[GETIDX(8)], -1);
    ASSERT_EQ(molatom[GETIDX(9)], 0);
    ASSERT_EQ(molatom[GETIDX(10)], 1);
    ASSERT_EQ(molatom[GETIDX(11)], 2);
    ASSERT_EQ(molatom[GETIDX(12)], 0);
    ASSERT_EQ(molatom[GETIDX(13)], 1);
    ASSERT_EQ(molatom[GETIDX(14)], 2);
    ASSERT_EQ(molatom[GETIDX(15)], -1);
    ASSERT_EQ(molatom[GETIDX(16)], -1);
}

TEST_F(AtomStyleTest, bond)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style bond");
    command("newton on");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "bond";
    expected.molecular     = Atom::MOLECULAR;
    expected.tag_enable    = 1;
    expected.molecule_flag = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;
    expected.has_bonds     = true;
    expected.has_nspecial  = true;
    expected.has_special   = true;
    expected.map_style     = 3;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box bond/types 2 "
            "extra/bond/per/atom 2 extra/special/per/atom 4");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("mass 3 4.4");
    command("bond_style zero");
    command("bond_coeff * 4.0");
    command("pair_coeff * *");
    command("create_bonds single/bond 1 1 5");
    command("create_bonds single/bond 1 1 3");
    command("create_bonds single/bond 2 3 5");
    command("create_bonds single/bond 2 3 6");
    command("create_bonds single/bond 2 5 6");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    // change sign on a couple of bond types. this will be undone on writing
    // a data file and thus should be positive again after reading it
    lmp->atom->bond_type[GETIDX(1)][0] *= -1;
    lmp->atom->bond_type[GETIDX(5)][0] *= -1;

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("units real");
    command("newton off");
    command("atom_style bond");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    command("bond_coeff * 4.0");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto num_bond  = lmp->atom->num_bond;
    auto bond_type = lmp->atom->bond_type;
    auto bond_atom = lmp->atom->bond_atom;

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 3);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 3);
    ASSERT_EQ(num_bond[GETIDX(6)], 2);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(5)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(6)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(6)][1], 5);

    BEGIN_HIDE_OUTPUT();
    command("clear");
    command("units real");
    command("atom_style bond");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    command("bond_coeff * 4.0");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x    = lmp->atom->x;
    auto v    = lmp->atom->v;
    auto type = lmp->atom->type;
    num_bond  = lmp->atom->num_bond;
    bond_type = lmp->atom->bond_type;
    bond_atom = lmp->atom->bond_atom;

    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);

    // change sign on bonds again. they will be toggled back in the restart
    lmp->atom->bond_type[GETIDX(1)][0] *= -1;
    lmp->atom->bond_type[GETIDX(5)][0] *= -1;

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    num_bond  = lmp->atom->num_bond;
    bond_atom = lmp->atom->bond_atom;
    bond_type = lmp->atom->bond_type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(num_bond[GETIDX(7)], 2);
    ASSERT_EQ(num_bond[GETIDX(9)], 2);
    ASSERT_EQ(num_bond[GETIDX(11)], 1);
    ASSERT_EQ(num_bond[GETIDX(12)], 0);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);
    ASSERT_EQ(bond_atom[GETIDX(7)][0], 11);
    ASSERT_EQ(bond_atom[GETIDX(7)][1], 9);
    ASSERT_EQ(bond_atom[GETIDX(9)][0], 11);
    ASSERT_EQ(bond_atom[GETIDX(9)][1], 12);
    ASSERT_EQ(bond_atom[GETIDX(11)][0], 12);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(7)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(7)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(9)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(9)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(11)][0], 2);

    BEGIN_HIDE_OUTPUT();
    command("delete_bonds all bond 2");
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    type = lmp->atom->type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 1);
    ASSERT_EQ(num_bond[GETIDX(5)], 2);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(num_bond[GETIDX(7)], 2);
    ASSERT_EQ(num_bond[GETIDX(8)], 1);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 4);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 4);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(4)][0], 2);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 8);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 7);
    ASSERT_EQ(bond_atom[GETIDX(7)][0], 8);
    ASSERT_EQ(bond_atom[GETIDX(7)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(8)][0], 6);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], -2);
    ASSERT_EQ(bond_type[GETIDX(4)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(7)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(7)][1], -2);
    ASSERT_EQ(bond_type[GETIDX(8)][0], -2);
}

TEST_F(AtomStyleTest, angle)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style angle");
    command("newton on");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style    = "angle";
    expected.molecular     = Atom::MOLECULAR;
    expected.tag_enable    = 1;
    expected.molecule_flag = 1;
    expected.has_type      = true;
    expected.has_mask      = true;
    expected.has_image     = true;
    expected.has_x         = true;
    expected.has_v         = true;
    expected.has_f         = true;
    expected.has_bonds     = true;
    expected.has_angles    = true;
    expected.has_nspecial  = true;
    expected.has_special   = true;
    expected.map_style     = 3;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box bond/types 2 angle/types 2 "
            "extra/bond/per/atom 2 extra/angle/per/atom 1 "
            "extra/special/per/atom 4");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("mass 3 4.4");
    command("pair_coeff * *");
    command("bond_style zero");
    command("bond_coeff * 4.0");
    command("angle_style zero");
    command("angle_coeff * 90.0");
    command("create_bonds single/bond 1 1 3");
    command("create_bonds single/bond 1 1 5");
    command("create_bonds single/bond 2 3 5");
    command("create_bonds single/bond 2 3 6");
    command("create_bonds single/bond 2 5 6");
    command("create_bonds single/angle 1 1 3 5");
    command("create_bonds single/angle 2 3 5 6");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangles, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    // change sign on a couple of angle types. this will be undone on writing
    // a data file and thus should be positive again after reading it
    lmp->atom->angle_type[GETIDX(3)][0] *= -1;
    lmp->atom->angle_type[GETIDX(5)][0] *= -1;

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("units real");
    command("newton off");
    command("atom_style angle");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("angle_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    command("bond_coeff * 4.0");
    command("angle_coeff * 90.0");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto num_bond    = lmp->atom->num_bond;
    auto bond_type   = lmp->atom->bond_type;
    auto bond_atom   = lmp->atom->bond_atom;
    auto num_angle   = lmp->atom->num_angle;
    auto angle_type  = lmp->atom->angle_type;
    auto angle_atom1 = lmp->atom->angle_atom1;
    auto angle_atom2 = lmp->atom->angle_atom2;
    auto angle_atom3 = lmp->atom->angle_atom3;

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 3);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 3);
    ASSERT_EQ(num_bond[GETIDX(6)], 2);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(5)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(6)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(6)][1], 5);
    ASSERT_EQ(num_angle[GETIDX(1)], 1);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 2);
    ASSERT_EQ(num_angle[GETIDX(4)], 0);
    ASSERT_EQ(num_angle[GETIDX(5)], 2);
    ASSERT_EQ(num_angle[GETIDX(6)], 1);
    ASSERT_EQ(angle_type[GETIDX(1)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][1], 2);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][1], 2);
    ASSERT_EQ(angle_type[GETIDX(6)][0], 2);
    ASSERT_EQ(angle_atom1[GETIDX(1)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(1)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(1)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(3)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(3)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(3)][1], 3);
    ASSERT_EQ(angle_atom2[GETIDX(3)][1], 5);
    ASSERT_EQ(angle_atom3[GETIDX(3)][1], 6);
    ASSERT_EQ(angle_atom1[GETIDX(5)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(5)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(5)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(5)][1], 3);
    ASSERT_EQ(angle_atom2[GETIDX(5)][1], 5);
    ASSERT_EQ(angle_atom3[GETIDX(5)][1], 6);
    ASSERT_EQ(angle_atom1[GETIDX(6)][0], 3);
    ASSERT_EQ(angle_atom2[GETIDX(6)][0], 5);
    ASSERT_EQ(angle_atom3[GETIDX(6)][0], 6);

    BEGIN_HIDE_OUTPUT();
    command("clear");
    command("units real");
    command("atom_style angle");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    command("bond_coeff * 4.0");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto type  = lmp->atom->type;
    num_bond   = lmp->atom->num_bond;
    bond_atom  = lmp->atom->bond_atom;
    num_angle  = lmp->atom->num_angle;
    angle_type = lmp->atom->angle_type;

    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);

    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(4)], 0);
    ASSERT_EQ(num_angle[GETIDX(5)], 1);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 2);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type       = lmp->atom->type;
    num_angle  = lmp->atom->num_angle;
    angle_type = lmp->atom->angle_type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(5)], 1);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(num_angle[GETIDX(7)], 0);
    ASSERT_EQ(num_angle[GETIDX(9)], 1);
    ASSERT_EQ(num_angle[GETIDX(11)], 1);
    ASSERT_EQ(num_angle[GETIDX(12)], 0);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 2);
    ASSERT_EQ(angle_type[GETIDX(9)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(11)][0], 2);

    BEGIN_HIDE_OUTPUT();
    command("delete_bonds all angle 2");
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    type = lmp->atom->type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(4)], 1);
    ASSERT_EQ(num_angle[GETIDX(5)], 0);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(num_angle[GETIDX(7)], 1);
    ASSERT_EQ(num_angle[GETIDX(8)], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(4)][0], -2);
    ASSERT_EQ(angle_type[GETIDX(7)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(8)][0], -2);
}

TEST_F(AtomStyleTest, full_ellipsoid)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    BEGIN_HIDE_OUTPUT();
    command("atom_style hybrid full ellipsoid");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style     = "hybrid";
    expected.molecular      = Atom::MOLECULAR;
    expected.tag_enable     = 1;
    expected.molecule_flag  = 1;
    expected.ellipsoid_flag = 1;
    expected.q_flag         = 1;
    expected.rmass_flag     = 1;
    expected.torque_flag    = 1;
    expected.angmom_flag    = 1;
    expected.has_type       = true;
    expected.has_mask       = true;
    expected.has_image      = true;
    expected.has_x          = true;
    expected.has_v          = true;
    expected.has_f          = true;
    expected.has_bonds      = true;
    expected.has_angles     = true;
    expected.has_dihedral   = true;
    expected.has_improper   = true;
    expected.has_nspecial   = true;
    expected.has_special    = true;
    expected.map_style      = 3;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    auto hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);

    BEGIN_HIDE_OUTPUT();
    command("create_box 3 box bond/types 2 "
            "extra/bond/per/atom 2 extra/special/per/atom 4");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("create_atoms 3 single  2.0  2.0 -2.1");
    command("create_atoms 3 single  2.0 -2.0  2.1");
    command("set type 1 mass 4.0");
    command("set type 2 mass 2.4");
    command("set type 3 mass 4.4");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("mass 3 4.4");
    command("set type 1 shape 1.0 1.0 1.0");
    command("set type 2 shape 3.0 0.8 1.1");
    command("set atom 1 quat 1.0 0.0 0.0  0.0");
    command("set atom 2 quat 0.0 1.0 0.0 90.0");
    command("set atom 3 quat 1.0 0.0 1.0 30.0");
    command("set atom 4 quat 1.0 1.0 1.0 60.0");
    command("set atom 1 charge -0.5");
    command("set atom 2 charge  0.5");
    command("set atom 3 charge -1.0");
    command("set atom 4 charge  1.0");
    command("set atom 5 charge  2.0");
    command("set atom 6 charge -2.0");
    command("bond_style zero");
    command("bond_coeff * 4.0");
    command("pair_coeff * *");
    command("create_bonds single/bond 1 1 5");
    command("create_bonds single/bond 1 1 3");
    command("create_bonds single/bond 2 3 5");
    command("create_bonds single/bond 2 3 6");
    command("create_bonds single/bond 2 5 6");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("units real");
    command("atom_style hybrid full ellipsoid");
    command("pair_style zero 4.0");
    command("bond_style zero");
    command("atom_modify map array");
    command("read_data test_atom_styles.data");
    command("pair_coeff * *");
    command("bond_coeff * 4.0");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x         = lmp->atom->x;
    auto v         = lmp->atom->v;
    auto q         = lmp->atom->q;
    auto type      = lmp->atom->type;
    auto ellipsoid = lmp->atom->ellipsoid;
    auto rmass     = lmp->atom->rmass;

    auto avec  = (AtomVecEllipsoid *)hybrid->styles[1];
    auto bonus = avec->bonus;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(5)][2], -2.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(6)][2], 2.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(5)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(6)][2], 0.0, EPSILON);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], 0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(4)], 1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(5)], 2.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(6)], -2.0, EPSILON);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(4)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], sqrt(0.5), EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    command("read_restart test_atom_styles.restart");
    command("replicate 1 1 2 bbox");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::MOLECULAR);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)hybrid->styles[1];
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(9)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(11)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(12)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(9)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(11)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(12)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.25056280708573159, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    hybrid    = (AtomVecHybrid *)lmp->atom->avec;
    avec      = (AtomVecEllipsoid *)hybrid->styles[1];
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(4)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(8)], -1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(5)], 4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(6)], 4.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(7)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(8)], 4.4, EPSILON);

    EXPECT_NEAR(bonus[0].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[0].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[1].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[1].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[2].shape[0], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[1], 0.5, EPSILON);
    EXPECT_NEAR(bonus[2].shape[2], 0.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[0], 1.5, EPSILON);
    EXPECT_NEAR(bonus[3].shape[1], 0.4, EPSILON);
    EXPECT_NEAR(bonus[3].shape[2], 0.55, EPSILON);
    EXPECT_NEAR(bonus[0].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[0].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[1].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[1].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[1].quat[3], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[2].quat[0], 1.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[1], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[2].quat[3], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[0], 0.9351131265310294, EPSILON);
    EXPECT_NEAR(bonus[3].quat[1], 0.25056280708573159, EPSILON);
    EXPECT_NEAR(bonus[3].quat[2], 0.0, EPSILON);
    EXPECT_NEAR(bonus[3].quat[3], 0.25056280708573159, EPSILON);
}

TEST_F(AtomStyleTest, property_atom)
{
    BEGIN_HIDE_OUTPUT();
    command("atom_modify map array");
    command("fix Properties all property/atom "
            "i_one d_two mol d_three q rmass i2_four 2 d2_five 3 ghost yes");
    END_HIDE_OUTPUT();

    AtomState expected;
    expected.atom_style         = "atomic";
    expected.molecular          = Atom::ATOMIC;
    expected.tag_enable         = 1;
    expected.map_style          = 1;
    expected.map_user           = 1;
    expected.has_type           = true;
    expected.has_image          = true;
    expected.has_mask           = true;
    expected.has_x              = true;
    expected.has_v              = true;
    expected.has_f              = true;
    expected.has_ivname         = true;
    expected.has_dvname         = true;
    expected.has_ianame         = true;
    expected.has_daname         = true;
    expected.has_extra_grow     = true;
    expected.has_extra_restart  = true;
    expected.has_extra_border   = true;
    expected.molecule_flag      = 1;
    expected.q_flag             = 1;
    expected.rmass_flag         = 1;
    expected.nivector           = 1;
    expected.ndvector           = 2;
    expected.nextra_grow        = 1;
    expected.nextra_grow_max    = 1;
    expected.nextra_restart     = 1;
    expected.nextra_restart_max = 1;
    expected.nextra_border      = 1;
    expected.nextra_border_max  = 1;
    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);

    BEGIN_HIDE_OUTPUT();
    command("create_box 2 box");
    command("create_atoms 1 single -2.0  2.0  0.1");
    command("create_atoms 1 single -2.0 -2.0 -0.1");
    command("create_atoms 2 single  2.0  2.0 -0.1");
    command("create_atoms 2 single  2.0 -2.0  0.1");
    command("mass 1 4.0");
    command("mass 2 2.4");
    command("set atom 1 charge -0.5");
    command("set atom 2 charge  0.5");
    command("set atom 3 charge -1.0");
    command("set atom 4 charge  1.0");
    command("set type 1 mass 4.0");
    command("set type 2 mass 2.4");
    command("set atom 1 mol 1");
    command("set atom 2 mol 2");
    command("set atom 3 mol 2");
    command("set atom 4 mol 1");
    command("set atom * i_one -5");
    command("set atom 3* i_one 5");
    command("set type 1 d_two 2.0");
    command("set type 2 d_two 1.0");
    command("set atom 1 d_three -2.5");
    command("set atom 2 d_three -1.0");
    command("set atom 3 d_three 0.5");
    command("set atom 4 d_three 2.0");
    command("set atom * d2_five[1] -5.9");
    command("set atom * d2_five[2] 1.1e-2");
    command("set atom * d2_five[3] .1");
    command("set atom 1*2 i2_four[1] -2");
    command("set atom 3*4 i2_four[1] -1");
    command("set atom * i2_four[2] 2");
    END_HIDE_OUTPUT();
    expected.natoms           = 4;
    expected.nlocal           = 4;
    expected.map_tag_max      = 4;
    expected.nmax             = 16384;
    expected.ntypes           = 2;
    expected.has_mass         = true;
    expected.has_mass_setflag = true;
    expected.has_sametag      = true;
    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_NE(lmp->atom->rmass, nullptr);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("write_data test_atom_styles.data nocoeff");
    command("clear");
    command("atom_style atomic");
    command("pair_style zero 4.0");
    command("units real");
    command("atom_modify map array");
    command("fix props all property/atom i_one d_two mol d_three q rmass "
            "i2_four 2 d2_five 3 ghost yes");
    command("read_data test_atom_styles.data fix props NULL Properties");
    command("pair_coeff * *");
    END_HIDE_OUTPUT();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, Atom::ATOMIC);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, Atom::MAP_ARRAY);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    auto x = lmp->atom->x;
    auto v = lmp->atom->v;
    auto q = lmp->atom->q;
    EXPECT_NEAR(x[GETIDX(1)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(1)][2], 0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][0], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(2)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][1], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(3)][2], -0.1, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][0], 2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][1], -2.0, EPSILON);
    EXPECT_NEAR(x[GETIDX(4)][2], 0.1, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(1)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(2)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(3)][2], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][0], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][1], 0.0, EPSILON);
    EXPECT_NEAR(v[GETIDX(4)][2], 0.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], 0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -1.0, EPSILON);
    EXPECT_NEAR(q[GETIDX(4)], 1.0, EPSILON);

    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);

    auto rmass = lmp->atom->rmass;
    auto one   = lmp->atom->ivector[0];
    auto two   = lmp->atom->dvector[0];
    auto three = lmp->atom->dvector[1];

    EXPECT_NEAR(rmass[GETIDX(1)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(4)], 2.4, EPSILON);
    EXPECT_EQ(one[GETIDX(1)], -5);
    EXPECT_EQ(one[GETIDX(2)], -5);
    EXPECT_EQ(one[GETIDX(3)], 5);
    EXPECT_EQ(one[GETIDX(4)], 5);
    EXPECT_NEAR(two[GETIDX(1)], 2.0, EPSILON);
    EXPECT_NEAR(two[GETIDX(2)], 2.0, EPSILON);
    EXPECT_NEAR(two[GETIDX(3)], 1.0, EPSILON);
    EXPECT_NEAR(two[GETIDX(4)], 1.0, EPSILON);
    EXPECT_NEAR(three[GETIDX(1)], -2.5, EPSILON);
    EXPECT_NEAR(three[GETIDX(2)], -1.0, EPSILON);
    EXPECT_NEAR(three[GETIDX(3)], 0.5, EPSILON);
    EXPECT_NEAR(three[GETIDX(4)], 2.0, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("pair_coeff * *");
    command("group two id 2:4:2");
    command("delete_atoms group two compress no");
    command("write_restart test_atom_styles.restart");
    command("clear");
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    command("read_restart test_atom_styles.restart");
    command("fix props all property/atom i_one d_two mol d_three q rmass "
            "i2_four 2 d2_five 3 ghost yes");
    END_HIDE_OUTPUT();
    expected.natoms           = 2;
    expected.nlocal           = 2;
    expected.nghost           = 0;
    expected.map_tag_max      = 3;
    expected.nmax             = 16384;
    expected.ntypes           = 2;
    expected.has_mass         = true;
    expected.has_mass_setflag = true;
    expected.has_sametag      = true;
    expected.has_extra        = true;
    expected.has_ivname       = true;
    expected.has_dvname       = true;
    expected.has_ianame       = true;
    expected.has_daname       = true;
    expected.nextra_store     = 12;

    ASSERT_ATOM_STATE_EQ(lmp->atom, expected);
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);

    q = lmp->atom->q;
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(3)], -1.0, EPSILON);

    rmass = lmp->atom->rmass;
    one   = lmp->atom->ivector[0];
    two   = lmp->atom->dvector[0];
    three = lmp->atom->dvector[1];
    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(3)], 2.4, EPSILON);
    EXPECT_EQ(one[GETIDX(1)], -5);
    EXPECT_EQ(one[GETIDX(3)], 5);
    EXPECT_NEAR(two[GETIDX(1)], 2.0, EPSILON);
    EXPECT_NEAR(two[GETIDX(3)], 1.0, EPSILON);
    EXPECT_NEAR(three[GETIDX(1)], -2.5, EPSILON);
    EXPECT_NEAR(three[GETIDX(3)], 0.5, EPSILON);

    BEGIN_HIDE_OUTPUT();
    command("reset_atom_ids");
    command("change_box all triclinic");
    END_HIDE_OUTPUT();
    ASSERT_EQ(lmp->atom->map_tag_max, 2);
    q = lmp->atom->q;
    EXPECT_NEAR(q[GETIDX(1)], -0.5, EPSILON);
    EXPECT_NEAR(q[GETIDX(2)], -1.0, EPSILON);

    rmass = lmp->atom->rmass;
    one   = lmp->atom->ivector[0];
    two   = lmp->atom->dvector[0];
    three = lmp->atom->dvector[1];
    EXPECT_NEAR(lmp->atom->mass[1], 4.0, EPSILON);
    EXPECT_NEAR(lmp->atom->mass[2], 2.4, EPSILON);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    EXPECT_NEAR(rmass[GETIDX(1)], 4.0, EPSILON);
    EXPECT_NEAR(rmass[GETIDX(2)], 2.4, EPSILON);
    EXPECT_EQ(one[GETIDX(1)], -5);
    EXPECT_EQ(one[GETIDX(2)], 5);
    EXPECT_NEAR(two[GETIDX(1)], 2.0, EPSILON);
    EXPECT_NEAR(two[GETIDX(2)], 1.0, EPSILON);
    EXPECT_NEAR(three[GETIDX(1)], -2.5, EPSILON);
    EXPECT_NEAR(three[GETIDX(2)], 0.5, EPSILON);
}

} // namespace LAMMPS_NS

int main(int argc, char **argv)
{
    MPI_Init(&argc, &argv);
    ::testing::InitGoogleMock(&argc, argv);

    // handle arguments passed via environment variable
    if (const char *var = getenv("TEST_ARGS")) {
        auto env = split_words(var);
        for (auto arg : env) {
            if (arg == "-v") {
                verbose = true;
            }
        }
    }
    if ((argc > 1) && (strcmp(argv[1], "-v") == 0)) verbose = true;

    int rv = RUN_ALL_TESTS();
    MPI_Finalize();
    return rv;
}
