package org.jboss.cache.marshall;

import org.jboss.cache.commands.ReplicableCommand;
import org.jboss.cache.factories.ComponentRegistry;
import org.jboss.cache.interceptors.InterceptorChain;
import org.jboss.cache.invocation.InvocationContextContainer;
import org.jgroups.Channel;
import org.jgroups.MembershipListener;
import org.jgroups.Message;
import org.jgroups.MessageListener;
import org.jgroups.blocks.RpcDispatcher;

/**
 * Extends {@link org.jgroups.blocks.RpcDispatcher} and adds the possibility that the marshaller may throw {@link org.jboss.cache.marshall.InactiveRegionException}s.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani</a>
 * @since 2.0.0
 */
public class InactiveRegionAwareRpcDispatcher extends CommandAwareRpcDispatcher
{
   org.jboss.cache.marshall.Marshaller requestMarshaller;

   /**
    * Only provide the flavour of the {@link RpcDispatcher} constructor that we care about.
    */
   public InactiveRegionAwareRpcDispatcher(Channel channel, MessageListener l, MembershipListener l2, Object serverObj,
                                           InvocationContextContainer container, InterceptorChain interceptorChain,
                                           ComponentRegistry componentRegistry)
   {
      super(channel, l, l2, serverObj, container, interceptorChain, componentRegistry);
   }

   @Override
   public void setRequestMarshaller(Marshaller m)
   {
      super.setRequestMarshaller(m);
      requestMarshaller = (org.jboss.cache.marshall.Marshaller) m;
   }


   /**
    * Message contains MethodCall. Execute it against *this* object and return result.
    * Use MethodCall.invoke() to do this. Return result.
    */
   @Override
   public Object handle(Message req)
   {
      if (isValid(req))
      {
         RegionalizedMethodCall rmc;
         ReplicableCommand command;

         try
         {
            // we will ALWAYS be using the marshaller to unmarshall requests.
            rmc = requestMarshaller.regionalizedMethodCallFromByteBuffer(req.getBuffer());
            command = rmc.command;
         }
         catch (Throwable e)
         {
            if (e instanceof InactiveRegionException)
            {
               if (trace) log.trace("Exception from marshaller: " + e.getMessage());
               return null;
            }

            if (trace) log.error("exception unmarshalling object", e);
            return e;
         }

         try
         {
            Object retVal = executeCommand(command, req);
            return new RegionalizedReturnValue(retVal, rmc);
         }
         catch (Throwable x)
         {
            if (trace) log.trace("Problems invoking command", x);
            return x;
         }
      }
      else
      {
         return null;
      }
   }
}
