/* dl.hh - load dynamic libraries
 * Copyright 2005 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_DL_HH
#define SHEVEK_DL_HH

#include <string>
#include "dlfcn.h"
#include "refbase.hh"
#include "error.hh"

namespace shevek
{
  /// Load symbols from dynamic libraries.
  /** Usage: create, open, get, get, ..., close
   */
  class dl : public refbase
  {
    void *m_handle;
  public:
    /// Create a new dl object.
    static Glib::RefPtr <dl> create () { return Glib::RefPtr <dl> (new dl ()); }
    /// Close object and free structures.
    ~dl ();
    /// Open a shared library.
    void open (std::string const &file = std::string ());
    /// Close the library, freeing the resources.
    /** This is done automatically if open is called again, or the object is destroyed.
     */
    void close ();
    /// Get a symbol from the library.
    /** Its type must be given by the caller and cannot be checked for correctness.
     */
    template <typename T> T &get (std::string const &name);
  protected:
    dl ();
  };

  template <typename T> T &dl::get (std::string const &name)
  {
    if (!m_handle)
      shevek_error ("unable to get symbol: no valid handle");
    union
    {
      void *input;
      T *output;
    } hack;
    hack.input = dlsym (m_handle, name.c_str ());
    if (!hack.input)
      shevek_error (ostring ("unable to get symbol: %s", dlerror ()));
    return *hack.output;
  }
}

#endif
