(*
 * Copyright (c) 2009-2015 Anil Madhavapeddy <anil@recoil.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* Convert an Apache mime.types file into a static ML data structure. *)

open Printf
open Str

let files = ref []

let extensions = ref true

let options = [
  "--files", Arg.Clear extensions,
    "Generate function mapping on filenames rather than on extensions" ;
  ]

let file_iter fn c =
  try while true do fn (input_line c) done
  with End_of_file -> ()

let _ =
  Arg.parse options (fun s -> files := s :: !files)
    (Printf.sprintf "Usage: %s [options]\nwhere options are:" Sys.argv.(0));
  let fun_name = if !extensions then "map_extension" else "map_file" in
  let dup = Hashtbl.create 101 in
  printf "(* This function is autogenerated by: %s *)\n" (String.concat " " (Array.to_list Sys.argv));
  printf "let %s = function\n" fun_name;
  List.iter (fun fname ->
    let fin = open_in fname in
    file_iter (fun l ->
      match split (regexp "\t+") l with
      | [] -> ()
      | [_] -> ()
      | [mime;_] when String.length mime > 0 && mime.[0] = '#' -> ()
      | [mime;exts] ->
        List.iter (fun e ->
            if not (Hashtbl.mem dup e) then begin
              printf " | \"%s\" -> \"%s\"\n" (String.escaped e) (String.escaped mime);
              Hashtbl.add dup e ()
            end
          ) (split (regexp " +") exts)
      | _ -> ()
    ) fin;
    close_in fin
  ) (List.rev !files);
  printf " | _ -> \"application/octet-stream\"\n%!"
