/*
 * lib.c         Generic functions.
 *
 * Version:      @(#)lib.c  1.36  13-Jan-1998  miquels@cistron.nl
 *
 */

#include "server.h"
#include <radiusclient.h>

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <values.h>
#include <time.h>

static int thisport = -1; /* port number */
static int chat_timeout;       /* Chat timeout, in seconds */
static int chat_send_delay;    /* Send delay, in 10th seconds */

/* GetPortNo should be used by the rest of the code to get the
 * "current" port number.  ONLY main.c should call SetPortNo().
 */

int GetPortNo()
{
  return thisport;             /* declared static above */
}

void SetPortNo(int num)
{
  thisport = num;
}

int GetChatTimeout()
{
  return chat_timeout;         /* again, static to lib.c */
}

void SetChatTimeout(int num)
{
  chat_timeout = num;
}

int GetChatSendDelay()
{
  return chat_send_delay;      /* You guessed it */
}

void SetChatSendDelay(int num)
{
  chat_send_delay = num;
}

/*
 * Malloc and die if failed.
 */
void *xmalloc(int size)
{
  void *p;

  if((p = malloc(size)) == NULL)
  {
    nsyslog(LOG_ERR, "Virtual memory exhausted.\n");
    exit(1);
  }
  memset(p, 0, size);
  return p;
}

/*
 * Realloc and die if failed.
 */
void *xrealloc(void *ptr, int size)
{
  void *p;

  if((p = realloc(ptr, size)) == NULL)
  {
    nsyslog(LOG_ERR, "Virtual memory exhausted.\n");
    exit(1);
  }
  return p;
}

void xsleep(unsigned int seconds)
{
  struct timespec ts;

  ts.tv_nsec = 0;
  ts.tv_sec = seconds;
  nanosleep(&ts, NULL);
}

void xusleep(unsigned long usec)
{
  struct timespec ts;

  ts.tv_nsec = usec * 1000;
  ts.tv_sec = 0;
  nanosleep(&ts, NULL);
}

/*
 * Strdup and die if failed.
 */
char *xstrdup(const char *str)
{
  char *p;

  if((p = strdup(str)) == NULL)
  {
    nsyslog(LOG_ERR, "Virtual memory exhausted.\n");
    exit(1);
  }
  return p;
}

/*
 * Signal functions.
 */
void block(int sig)
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, sig);
  sigprocmask(SIG_BLOCK, &set, NULL);
}
void unblock(int sig)
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, sig);
  sigprocmask(SIG_UNBLOCK, &set, NULL);
}

/* Check for the existance of a device node and return the absolute name
 * returned by dereferencing all sym-links. */
char *check_device(const char *file)
{
  char *real_name;

  if(file[0] != '/')
  {
    char *str = xmalloc(strlen(file) + 6);
    strcpy(str, "/dev/");
    strcat(str, file);
    real_name = canonicalize_file_name(str);
    free(str);
  }
  else
    real_name = canonicalize_file_name(file);

  return real_name;
}

/*
 *  Print number in static buffer (itoa).
 */

char *static_num(int i)
{
  static char buf[16];

  snprintf(buf, sizeof (buf), "%d", i);
  return buf;
}

/*
 *  Print dotted IP address.
 */
const char *dotted(unsigned int ipno)
{
  struct in_addr ia;

  ia.s_addr = ipno;
  return inet_ntoa(ia);
}

in_port_t *get_port_ptr(struct sockaddr *sa)
{
#ifdef HAVE_IPV6
  if(sa->sa_family == AF_INET6)
    return &((struct sockaddr_in6 *)sa)->sin6_port;
#endif
  return &((struct sockaddr_in *)sa)->sin_port;
}

static const void *get_addr_ptr(const struct sockaddr *sa)
{
#ifdef HAVE_IPV6
  if(sa->sa_family == AF_INET6)
    return &((struct sockaddr_in6 *)sa)->sin6_addr;
#endif
  return &((struct sockaddr_in *)sa)->sin_addr;
}

const char *dotted_sa(const struct sockaddr *sa, bool port)
{
#ifdef HAVE_IPV6
  static char buf[INET6_ADDRSTRLEN + 8];
#else
  static char buf[INET_ADDRSTRLEN + 8];
#endif
  char *ptr = buf;
  if(port)
  {
    *ptr = '[';
    ptr++;
  }
  if(!inet_ntop(sa->sa_family, get_addr_ptr(sa), ptr, sizeof(buf) - 2))
  {
    return "address error";
  }
  if(!port)
    return buf;
  ptr += strlen(ptr);
  snprintf(ptr, sizeof(buf) - (ptr - buf), "]%d", ntohs(*get_port_ptr((struct sockaddr *)sa)) );
  buf[sizeof(buf) - 1] = '\0';
  return buf;
}

int sa_len(const struct sockaddr *sa)
{
#ifdef HAVE_IPV6
  if(sa->sa_family == AF_INET6)
    return sizeof(struct sockaddr_in6);
#endif
  return sizeof(struct sockaddr_in);
}

/*
 * Strips login name and writes it to buffer.
 */

static void strip_login(char *buffer, int buf_size, const char *login)
{
  char *ptr;

  if(strchr("PCS!L", login [0]))
  {
    snprintf (buffer, buf_size, "%s", login+1);
  }
  else
  {
    snprintf (buffer, buf_size, "%s", login);
    ptr = strrchr (buffer, '.');
    if(ptr)
    {
      if(!strcmp(ptr, ".slip") || !strcmp(ptr, ".cslip") || !strcmp(ptr, ".ppp") )
        *ptr = 0;
    }
  }
}

int get_sessiontime(const struct auth *ai)
{
  int limit;
  if(!lineconf.login_time_limited)
    return ai->sessiontime;
  limit = chktimes() * 60;
  if(!ai->sessiontime || limit > ai->sessiontime)
    return limit;
  return ai->sessiontime;
}

/*
 *
 * Expands a format string
 *
 * %l: login name
 * %L: stripped login name
 * %p: NAS port number
 * %P: protocol
 * %b: port speed
 * %H: host for telnet/ssh connections
 * %i: local IP
 * %j: remote IP
 * %1: first byte (MSB) of remote IP
 * %2: second byte of remote IP
 * %3: third byte of remote IP
 * %4: fourth (LSB) byte of remote IP
 * %c: connect-info
 * %m: netmask
 * %M: "multilink" if the RADIUS server has PW_NAS_PORT_LIMIT set to > 1
 * %t: MTU
 * %r: MRU
 * %I: idle timeout
 * %T: session timeout
 * %h: hostname
 * %g: PID
 * %S: session timeout (other versions of Portslave implement this).
 * %d: DCD setting, "modem" if dcd is true, "local" if false.
 * %%: %
 *
 * Everything else is copied unchanged.
 *
 */

void expand_format(char *buffer, size_t size, const char *format, const struct auth *ai)
{
  char ch;
  int done;
  char str [256];
  unsigned long int byte_val;

  if(size-- == 0)
    return;
  while(size)
  {
    if((ch = *format++) == '%')
    {
      switch (*format)
      {
        case 'l':
          snprintf (str, sizeof(str), "%s", ai->login);
        break;
        case 'L':
          strip_login (str, sizeof (str), ai->login);
        break;
        case 'p':
          snprintf (str, sizeof(str), "%03d", ai->nasport);
        break;
        case 'P':
          snprintf (str, sizeof(str), "%c", (char) ai->proto);
        break;
        case 'b':
          snprintf (str, sizeof(str), "%d", lineconf.speed);
        break;
        case 'H':
          if(ai->host)
          snprintf (str, sizeof(str), "%s", dotted(ai->host));
        break;
        case 'i':
          snprintf(str, sizeof(str), "%s", dotted(lineconf.loc_host));
        break;
        case 'j':
          snprintf (str, sizeof(str), "%s", dotted(ai->address));
        break;
        case '1':
          byte_val = (ntohl (ai->address) >> (3*BITSPERBYTE)) & 0xFF;
          snprintf (str, sizeof(str), "%ld", byte_val);
        break;
        case '2':
          byte_val = (ntohl (ai->address) >> (2*BITSPERBYTE)) & 0xFF;
          snprintf (str, sizeof(str), "%ld", byte_val);
        break;
        case '3':
          byte_val = (ntohl (ai->address) >> BITSPERBYTE) & 0xFF;
          snprintf (str, sizeof(str), "%ld", byte_val);
        break;
        case '4':
          byte_val = ntohl (ai->address) & 0xFF;
          snprintf (str, sizeof(str), "%ld", byte_val);
        break;
        case 'c':
          snprintf (str, sizeof(str), "%s", ai->conn_info);
        break;
        case 'm':
          snprintf (str, sizeof(str), "%s", dotted(ai->netmask));
        break;
        case 'M':
          snprintf (str, sizeof(str), "%s", (ai->port_limit > 1) ? "multilink" : "");
        break;
        case 't':
          snprintf (str, sizeof(str), "%d", ai->mtu);
        break;
        case 'r':
          snprintf (str, sizeof(str), "%d", ai->mru);
        break;
        case 'I':
          snprintf (str, sizeof(str), "%d", ai->idletime);
        break;
        case 'S':
        case 'T':
          snprintf (str, sizeof(str), "%d", get_sessiontime(ai));
        break;
        case 'h':
          snprintf(str, sizeof(str), "%s", lineconf.hostname);
        break;
        case 'g':
          snprintf(str, sizeof(str), "%d", getpid());
        break;
        case 'd':
          snprintf(str, sizeof(str), "%s", lineconf.dcd ? "modem" : "local");
        break;
        case '%':
          snprintf (str, sizeof(str), "%c", '%');
        break;
        default:
          *buffer++ = '?';
          --size;
          continue;
      }
      ++format;
      if((done = snprintf (buffer, size+1, "%s", str)) == -1)
        return;
      buffer += done;
      size -= done;
    }
    else
    {
      if((*buffer = ch) == '\0')
        return;
      ++buffer;
      --size;
    }
  }
  *buffer = '\0';
}

#ifdef PORTSLAVE_CLIENT_IP_RULES
/*
 * Convert the valid ip string into the list of rules.
 *
 *   Multiple rules may reside in the string, separated by commas.
 *
 * NOTES:
 *
 *     - The previous contents of the auth structure for IP rules will be
 *       ignored and lost.
 */

static void conv_ip_rules(struct auth *ai, const char *rule_str)
{
  int             num_rule;
  int             cur;
  const char      *ptr;
  const char      *start;

  /* If the rule string is empty, just return now. */
  if(rule_str == NULL || rule_str[0] == '\0')
    return;

  /* Count the number of rules in order to size the array. */

  num_rule = 1;
  ptr = rule_str;

  while(ptr != NULL)
  {
    ptr = strchr(ptr, ',');

    if(ptr != NULL)
    {
      num_rule++;
      ptr++;
    }
  }

  /* Allocate the memory for the list of strings. */
  ai->valid_ip_rules = xmalloc(sizeof (char *) * num_rule);
  ai->num_ip_rules      = num_rule;

  /* Save each rule */

  ptr  = rule_str;
  start = rule_str;
  cur  = 0;

  /* Loop until the last rule is copied.  Check the size */
  /*  of the array for robustness; this should not be    */
  /*  necessary.                                         */

  while(ptr != NULL && cur < num_rule)
  {
    char    *one_rule;

    ptr = strchr(start, ',');

    /* If the end of the string was found, just save the */
    /*  string from the last saved point; otherwise,     */
    /*  save only the string up to the comma.            */

    if(ptr == NULL)
    {
      one_rule = xstrdup(start);
    }
    else
    {
      /* Check for the empty string. */
      if(ptr == start)
      {
        one_rule = xmalloc(sizeof (char));
        one_rule[0] = '\0';
      }
      else
      {
        size_t  len;

        /* Find the length of the string.  */
        /*  Note that ptr is one character */
        /*  past the end of the string.    */

        len = ptr - start;

        one_rule = xmalloc(sizeof (char) * (len + 1));

        strncpy(one_rule, start, len);
        one_rule[len] = '\0';
      }

      start = ptr + 1;
    }

    ai->valid_ip_rules[cur] = one_rule;
    cur++;
  }
}
#endif

/*
 *  Create a new session id.
 */
static char *rad_sessionid()
{
     char *id = xmalloc(13);
     snprintf(id, 13, "%08X%04X", (unsigned int)time(NULL), getpid() & 0xffff);
     return id;
}
	 
/*
 *      Fill in the auth struct for now.
 */
static void fill_auth_struct(struct auth *ai)
{
  memset(ai, 0, sizeof(struct auth));
  strcpy(ai->login, "NONE"); /* login is >4 chars long) */
  ai->nasport  = GetPortNo();
  ai->proto    = lineconf.protocol;
  ai->address  = lineconf.rem_host;
  ai->netmask  = lineconf.netmask;
  ai->mtu      = lineconf.mtu;
  ai->mru      = lineconf.mru;
  ai->porttype = lineconf.porttype;
  ai->localip  = lineconf.loc_host;
  /* first set the Acct-Session-Id to be used for auth and for acct */
  ai->acct_session_id = rad_sessionid();
  ai->start    = time(NULL);
    

  if(ai->netmask == 0)
    ai->netmask = 0xFFFFFFFF;

#ifdef PORTSLAVE_CLIENT_IP_RULES
  conv_ip_rules(ai, lineconf.valid_ip);
#endif
}

/**********************************************************************
* %FUNCTION: radius_init
* %ARGUMENTS:
*  msg -- buffer of size BUF_LEN for error message
* %RETURNS:
*  negative on failure; non-negative on success
* %DESCRIPTION:
*  Initializes radiusclient library
***********************************************************************/
static int
radius_init()
{
    if(rc_read_config(lineconf.radclient_config_file) != 0) {
        nsyslog(LOG_ERR, "RADIUS: Can't read config file %s",
                 lineconf.radclient_config_file);
        return -1;
    }

    if (rc_read_dictionary(rc_conf_str("dictionary")) != 0) {
        nsyslog(LOG_ERR, "RADIUS: Can't read dictionary file %s",
                 rc_conf_str("dictionary"));
        return -1;
    }

    if (rc_read_mapfile(rc_conf_str("mapfile")) != 0)   {
        nsyslog(LOG_ERR, "RADIUS: Can't read map file %s",
                 rc_conf_str("mapfile"));
        return -1;
    }
    return 0;
}

/*
 *  Initialize struct and read config file.
 */
int rad_init(const char *config_file, int port, struct auth *ai, const char *tty)
{
  /*
   *  Read the config file.
   */
  initcfg();
  SetPortNo(port);
  if(readcfg(config_file, tty) < 0)
    return -1;

  if(GetPortNo() < 0)
  {
    nsyslog(LOG_ERR, "\"%s\": not in config file", tty);
    return -1;
  }

  /*
   *  Fill in the auth struct for now.
   */
  fill_auth_struct(ai);

  if(ai->proto == P_PPP_ONLY)
    ai->proto = P_AUTOPPP;

  if(radius_init() != 0)
    return -1;

  return GetPortNo();
}

/* when access is denied it returns a negative number indicating the amount of
   time in minutes until a call is permitted.
   When access is allowed it will return 0 if the call can have unrestricted
   duration, or a positive number specifying the length of the call in minutes
   if there is a limit to the length.
   Currently does not coalesce times.  So if today we don't allow calls after
   2000 and tomorrow we don't allow them before 0800 then at 2200 it will sleep
   for 2 hours and then sleep for 8 hours.  However if you limit the call time
   based on the time allowed then it will give the wrong time unless you make
   sure that every allowed time is part of exactly one range.
*/
long chktimes()
{
  time_t time_now;
  struct tm now;
  int now_hhmm;
  struct time_ent *te = &lineconf.login_time[0];
  int rc = -1440;

  if(!te || te->days == 0)
    return 0; /* skip time check if not defined or malformed time definition */
  time_now = time(NULL);  /* get current time */
  now = *(localtime(&time_now));  /* Break it into bits */
  now_hhmm = now.tm_hour * 60 + now.tm_min;

  while(te->days)
  {
    if((1 << now.tm_wday) & lineconf.login_time->days)
    {   /* Date within range */
      if(now_hhmm >= te->start_time && now_hhmm <= te->end_time)
      {
        if(lineconf.login_time_limited)
          return te->end_time - now_hhmm;
        return 0;
      }
      if(now_hhmm < te->start_time)
      {
        int tmp = now_hhmm - te->start_time;
        if(tmp > rc)
          rc = tmp;
      }
    }
    te++;
  }
  return rc;
}

