% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connections.R
\name{conn_create_fifo}
\alias{conn_create_fifo}
\alias{conn_connect_fifo}
\title{Processx FIFOs}
\usage{
conn_create_fifo(
  filename = NULL,
  read = NULL,
  write = NULL,
  encoding = "",
  nonblocking = TRUE
)

conn_connect_fifo(
  filename,
  read = NULL,
  write = NULL,
  encoding = "",
  nonblocking = TRUE
)
}
\arguments{
\item{filename}{File name of the FIFO. On Windows it the name of the
pipe within the \verb{\\\\?\\pipe\\} namespace, either the full name, or the
part after that prefix. If \code{NULL}, then a random name
is used, on Unix in the R temporary directory: \code{\link[base:tempfile]{base::tempdir()}}.}

\item{read}{If \code{TRUE} then connect to the read end of the FIFO.
Exactly one of \code{read} and \code{write} must be set to \code{TRUE}.}

\item{write}{If \code{TRUE} then connect to the write end of the FIFO.
Exactly one of \code{read} and \code{write} must be set to \code{TRUE}.}

\item{encoding}{Encoding to assume.}

\item{nonblocking}{Whether this should be a non-blocking FIFO.
Note that blocking FIFOs are not well tested and might not work well with
\code{\link[=poll]{poll()}}, especially on Windows. We might remove this option in the
future and make all FIFOs non-blocking.}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Create a FIFO for inter-process communication
Note that these functions are currently experimental.
}
\details{
\code{conn_create_fifo()} creates a FIFO and connects to it.
On Unix this is a proper FIFO in the file system, in the R temporary
directory. On Windows it is a named pipe.

Use \code{\link[=conn_file_name]{conn_file_name()}} to query the name of the FIFO, and
\code{conn_connect_fifo()} to connect to the other end.

\code{conn_connect_fifo()} connects to a FIFO created with
\code{conn_create_fifo()}, typically in another process. \code{filename} refers
to the name of the pipe on Windows.

On Windows, \code{conn_connect_fifo()} may be successful even if the
FIFO does not exist, but then later \code{poll()} or read/write operations
will fail. We are planning on changing this behavior in the future,
to make \code{conn_connect_fifo()} fail immediately, like on Unix.
}
\section{Notes}{
\subsection{In general Unix domain sockets work better than FIFOs, so we suggest}{

you use sockets if you can. See \code{\link[=conn_create_unix_socket]{conn_create_unix_socket()}}.
}

\subsection{Creating the read end of the FIFO}{

This case is simpler. To wait for a writer to connect to the FIFO
you can use \code{\link[=poll]{poll()}} as usual. Then use \code{\link[=conn_read_chars]{conn_read_chars()}} or
\code{\link[=conn_read_lines]{conn_read_lines()}} to read from the FIFO, as usual. Use
\code{\link[=conn_is_incomplete]{conn_is_incomplete()}} \emph{after} a read to check if there is more data,
or the writer is done.
}

\subsection{Creating the write end of the FIFO}{

This is somewhat trickier. Creating the (non-blocking) FIFO does not
block. However, there is no easy way to tell if a reader is connected
to the other end of the FIFO or not. On Unix you can start using
\code{\link[=conn_write]{conn_write()}} to try to write to it, and this will succeed, until the
buffer gets full, even if there is no reader. (When the buffer is full
it will return the data that was not written, as usual.)

On Windows, using \code{\link[=conn_write]{conn_write()}} to write to a FIFO without a reader
fails with an error. This is not great, we are planning to improve it
later.

Right now, one workaround for this behavior is for the reader to
connunicate to the writer process independenctly that it has connected
to the FIFO. (E.g. another FIFO in the opposite direction can do that.)
}
}

\examples{
# Example for a non-blocking FIFO

# Need to open the reading end first, otherwise Unix fails
reader <- conn_create_fifo()

# Always use poll() before you read, with a timeout if you like.
# If you read before the other end of the FIFO is connected, then
# the OS (or processx?) assumes that the FIFO is done, and you cannot
# read anything.
# Now poll() tells us that there is no data yet.
poll(list(reader), 0)

writer <- conn_connect_fifo(conn_file_name(reader), write = TRUE)
conn_write(writer, "hello\nthere!\n")

poll(list(reader), 1000)
conn_read_lines(reader, 1)
conn_read_chars(reader)

conn_is_incomplete(reader)

close(writer)
conn_read_chars(reader)
conn_is_incomplete(reader)

close(reader)
}
\seealso{
\href{https://processx.r-lib.org/dev/articles/internals.html}{processx internals}
}
