use psa_crypto::operations::hash;
use psa_crypto::types::algorithm::Hash;

#[test]
fn hash_compute_ripemd160() {
    const MESSAGE: [u8; 3] = [0x61, 0x62, 0x63];
    const HASH: [u8; 20] = [
        0x8e, 0xb2, 0x08, 0xf7, 0xe0, 0x5d, 0x98, 0x7a, 0x9b, 0x04, 0x4a, 0x8e, 0x98, 0xc6, 0xb0,
        0x87, 0xf1, 0x5a, 0x0b, 0xfc,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Ripemd160;
    let mut hash = vec![0; hash_alg.hash_length()];
    let size = hash::hash_compute(hash_alg, &MESSAGE, &mut hash).unwrap();
    hash.resize(size, 0);
    assert_eq!(&HASH, &hash[..]);
}

#[test]
fn hash_compute_sha256() {
    const MESSAGE: [u8; 3] = [0xb0, 0xbd, 0x69];
    const HASH: [u8; 32] = [
        0x40, 0x96, 0x80, 0x42, 0x21, 0x09, 0x3d, 0xdc, 0xcf, 0xbf, 0x46, 0x83, 0x14, 0x90, 0xea,
        0x63, 0xe9, 0xe9, 0x94, 0x14, 0x85, 0x8f, 0x8d, 0x75, 0xff, 0x7f, 0x64, 0x2c, 0x7c, 0xa6,
        0x18, 0x03,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Sha256;
    let mut hash = vec![0; hash_alg.hash_length()];
    let size = hash::hash_compute(hash_alg, &MESSAGE, &mut hash).unwrap();
    hash.resize(size, 0);
    assert_eq!(&HASH, &hash[..]);
}

#[test]
fn hash_compare_sha256() {
    const MESSAGE: [u8; 3] = [0xb0, 0xbd, 0x69];
    const HASH: [u8; 32] = [
        0x40, 0x96, 0x80, 0x42, 0x21, 0x09, 0x3d, 0xdc, 0xcf, 0xbf, 0x46, 0x83, 0x14, 0x90, 0xea,
        0x63, 0xe9, 0xe9, 0x94, 0x14, 0x85, 0x8f, 0x8d, 0x75, 0xff, 0x7f, 0x64, 0x2c, 0x7c, 0xa6,
        0x18, 0x03,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Sha256;
    hash::hash_compare(hash_alg, &MESSAGE, &HASH).unwrap();
}

#[test]
fn hash_compare_ripemd160() {
    const MESSAGE: [u8; 3] = [0x61, 0x62, 0x63];
    const HASH: [u8; 20] = [
        0x8e, 0xb2, 0x08, 0xf7, 0xe0, 0x5d, 0x98, 0x7a, 0x9b, 0x04, 0x4a, 0x8e, 0x98, 0xc6, 0xb0,
        0x87, 0xf1, 0x5a, 0x0b, 0xfc,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Ripemd160;
    hash::hash_compare(hash_alg, &MESSAGE, &HASH).unwrap();
}

#[test]
fn hash_compare_sha256_fail() {
    const MESSAGE: [u8; 3] = [0xb0, 0xbd, 0x69];
    const HASH: [u8; 20] = [
        0x8e, 0xb2, 0x08, 0xf7, 0xe0, 0x5d, 0x98, 0x7a, 0x9b, 0x04, 0x4a, 0x8e, 0x98, 0xc6, 0xb0,
        0x87, 0xf1, 0x5a, 0x0b, 0xfc,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Sha256;
    hash::hash_compare(hash_alg, &MESSAGE, &HASH).unwrap_err();
}

#[test]
fn hash_compare_ripemd160_fail() {
    const MESSAGE: [u8; 3] = [0x61, 0x62, 0x63];
    const HASH: [u8; 32] = [
        0x40, 0x96, 0x80, 0x42, 0x21, 0x09, 0x3d, 0xdc, 0xcf, 0xbf, 0x46, 0x83, 0x14, 0x90, 0xea,
        0x63, 0xe9, 0xe9, 0x94, 0x14, 0x85, 0x8f, 0x8d, 0x75, 0xff, 0x7f, 0x64, 0x2c, 0x7c, 0xa6,
        0x18, 0x03,
    ];
    psa_crypto::init().unwrap();
    let hash_alg = Hash::Ripemd160;
    hash::hash_compare(hash_alg, &MESSAGE, &HASH).unwrap_err();
}
