/*
 *   Copyright (C) 2008, 2009 Nicolas Vion <nico@yojik.eu>
 *
 *   This file is part of Swac-scan.
 * 
 *   Swac-scan is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *   
 *   Swac-scan is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with Swac-scan.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"
#include "index.hh"

#include <iostream>

#include "ogg.hh"
#include "flac.hh"
#include "speex.hh"
#include "mp3.hh"
#include "stringutils.hh"


#ifdef HAVE_GIOMM

#include <giomm.h>
#include <glibmm.h>

void IndexTags::tags_import_dir(std::string const path, bool const verbose) {
	try {
		Glib::RefPtr<Gio::File> directory = Gio::File::create_for_path(path);
		Glib::RefPtr<Gio::FileEnumerator> enumerator = directory->enumerate_children();

		while (Glib::RefPtr<Gio::FileInfo> file_info = enumerator->next_file()) {
			std::string filename = Glib::build_filename(path, file_info->get_name());

			if (file_info->get_file_type() == Gio::FILE_TYPE_DIRECTORY and !file_info->is_symlink()) {
				if (options.recurse)
					tags_import_dir(filename, verbose);
			}
			else {
				std::string content_type = file_info->get_content_type();

				Tag *tag = NULL;
				if (content_type == "audio/x-vorbis+ogg")
					tag = new class TagOgg;
				else if (content_type == "audio/x-speex")
					tag = new class TagSpeex;
				else if (content_type == "audio/x-flac")
					tag = new class TagFlac;
				else if (content_type == "audio/mpeg")
					tag = new class TagMp3;
				
				if (tag != NULL) {
					tag->extract_from_file(filename, options.ignore_standards_fields);
					if (verbose) {
						export_tag(tag);
						delete(tag);
					}
					else
						tags.push_back(*tag);
				}
			}

			//patch that fix a bug of gtkmm
			//cf: http://bugzilla.gnome.org/show_bug.cgi?id=556387
			if (G_OBJECT(file_info->gobj())->ref_count > 1)
				file_info->unreference();
		}
		enumerator->close();
	}

	catch (const Glib::Exception& ex) {
		std::cerr << "Error: " << ex.what() << std::endl; 
	}
}


#else

#include <sys/types.h>
#include <sys/stat.h> 
#include <dirent.h>

#define CMP_EXT(str, ext) (str.size() > sizeof(ext) and str.substr(str.size() - sizeof(ext), sizeof(ext) + 1) == "." ext)

#define PATH_DELIM "/"

bool is_dir(std::string path) {
	struct stat stFileInfo;
	return (stat(path.c_str(), &stFileInfo) == 0) && S_ISDIR(stFileInfo.st_mode);
}

void IndexTags::tags_import_dir(std::string const path, bool const verbose) {
	DIR *dp;
	if ((dp = opendir((path != "") ? path.c_str() : "." PATH_DELIM)) == NULL) {
		std::cerr << "Can not open dir \"" << path << "\"" << std::endl;
		return;
	}
	
	dirent *d;
	while ((d = readdir(dp)) != NULL) {
		std::string filename(d->d_name);
		if (filename != "." and filename != "..") {
			std::string filepath(path + filename);

			if (is_dir(filepath)) {
				if (options.recurse)
					tags_import_dir(filepath + PATH_DELIM, verbose);
			}
			else {
				Tag *tag = NULL;
				if (CMP_EXT(filename, "ogg"))
					tag = new class TagOgg;
				else if (CMP_EXT(filename, "spx"))
					tag = new class TagSpeex;
				else if (CMP_EXT(filename, "flac"))
					tag = new class TagFlac;
				else if (CMP_EXT(filename, "mp3"))
					tag = new class TagMp3;

				if (tag != NULL) {
					tag->extract_from_file(filepath, options.ignore_standards_fields);
					if (verbose) {
						export_tag(tag);
						delete(tag);
					}
					else
						tags.push_back(*tag);
				}
			}
		}
	}
}
#endif

unsigned int IndexTags::count() {
	return tags.size();
}

void IndexTags::tags_print() {
	for (ListTags::iterator it = tags.begin(); it != tags.end(); ++it)
		it->print();
}

void IndexTags::constants_find() {
	ListTags::iterator tag = tags.begin();
	if (tag == tags.end())
		return;

	constants = tag->fields;

	for (tag++; tag != tags.end(); tag++) {
		//Check whether all fields value are egal the the constants
		for (ListFields::iterator field = tag->fields.begin(); field != tag->fields.end(); field++) {
			ListFields::iterator it = constants.find(field->first);
			if (it != constants.end() and it->second != field->second) {
				//std::cout << "field " << it->first << " is not constant. found value: " << it->second << " != " << field->second << std::endl; 
				constants.erase(it);
			}
		}
		//Check whether all constants are defined in this tag (if defined already checked upside)
		for (ListFields::iterator field = constants.begin(); field != constants.end(); field++) {
			if (tag->fields.find(field->first) == constants.end()) {
				std::cout << "field " << field->first << " is not constant. Not defined in " << tag->get_filename() << std::endl; 
				constants.erase(field);
			}
		}
	}
}

void IndexTags::constants_print() {
	for (ListFields::iterator field = constants.begin(); field != constants.end(); field++)
		std::cout << field->first << "=" << field->second << std::endl;
}

void IndexTags::export_head() {
	std::cout << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;
//	std::cout << "<!DOCTYPE index SYSTEM \"http://shtooka.net/swac/index.dtd\">" << std::endl;
	std::cout << "<index>" << std::endl;
	if (constants.size() > 0) {
		std::cout << "<group ";
		for (ListFields::iterator field = constants.begin(); field != constants.end(); field++)
			std::cout << std::endl << "\t" << field->first << "=\"" << StringUtils::strip_slashes(field->second) << "\"";
		std::cout << ">";
	}
}

void IndexTags::export_bottom() {
	if (constants.size() > 0)
		std::cout << std::endl << "</group>";
	std::cout << std::endl << "</index>" << std::endl;
}

void IndexTags::export_tag(Tag *tag) {
	std::cout << std::endl << "\t<file path=\"" << tag->get_filename() << "\"><tag ";
	for (ListFields::iterator field = tag->fields.begin(); field != tag->fields.end(); field++) {
		if (constants.find(field->first) == constants.end())
			std::cout << std::endl << "\t\t" << field->first << "=\"" << StringUtils::strip_slashes(field->second) << "\"";
	}
	std::cout << std::endl << "\t/></file>";
}

void IndexTags::export_all() {
	export_head();
	for (ListTags::iterator tag = tags.begin(); tag != tags.end(); tag++) {
		export_tag(&(*tag));
	}
	export_bottom();
}

IndexTags::~IndexTags() {
}

void IndexTagsIni::export_head() {
	if (constants.size()) {
		std::cout << "[GLOBAL]";
		for (ListFields::iterator field = constants.begin(); field != constants.end(); field++)
			std::cout << std::endl << StringUtils::upper(field->first) << "=" << field->second;
		std::cout << std::endl;
	}	
}

void IndexTagsIni::export_bottom() {
	std::cout << std::endl;
}

void IndexTagsIni::export_tag(Tag *tag) {
	std::cout << std::endl << "[" << tag->get_filename() << "]";
	for (ListFields::iterator field = tag->fields.begin(); field != tag->fields.end(); field++) {
		if (constants.find(field->first) == constants.end())
			std::cout << std::endl << StringUtils::upper(field->first) << "=" << field->second;
	}
	std::cout << std::endl;
}

